const app = require('../app');
const error = require('../error');

const BasicCollection = require('./BasicCollection');
const ClientFilterModel = require('../model/clients/ClientFilterModel');

/**
 * @class ClientsCollection
 * @extends BasicCollection
 * @extends Backbone.Events
 */
const ClientsCollection = BasicCollection.extend({

    url: '/api/v1.0/client',

    model: require('../model/clients/ClientModel'),

    initialize(models, options) {
        this.mapQueryOptions(options);

        this.setFilterParams = this.setFilterParams.bind(this);

        this.listenTo(this, 'change:filterparams:preset', this.setFilterParams);

        this.options = _.extend({}, this.options, options.options);
    },

    options: {
        page: 1,
        total: 1,
        pageCapacity: 100500
    },

    parse(response) {
        return response.items;
    },

    getSortGroups() {
        const sortGroups = {
            dc: [],
            os: [],
            ram: [],
            cpu: [],
            ncpu: [],
            type: [],
            owner: []
        };

        this.each(function (client) {
            const props = client.getGroupableProps();

            Object.keys(props).forEach(propName => {
                if (props[propName]) {
                    const propValues = (Array.isArray(props[propName]) ? props[propName] : [props[propName]]);

                    propValues.forEach(propValue => {
                        if (!sortGroups[propName].includes(propValue)) {
                            sortGroups[propName].push(propValue);
                        }
                    });
                }
            }, this);
        });

        sortGroups.ram = this.compactifyRamGroup(sortGroups.ram);
        sortGroups.tags = this.compactifyTagsGroup();

        return sortGroups;
    },

    compactifyTagsGroup() {
        const tags = {};

        this.options._tags.forEach(item => {
            tags[item.get('name')] = item.get('items');
        });

        return tags;
    },

    compactifyRamGroup(ramGroup) {
        return _.uniq(ramGroup
            .sort((a, b) => {
                return b - a;
            })
            .map(value => {
                return ('~' + Math.floor(value / 1024 / 1024 / 1024) + ' GB');
            }));
    },

    activateAutoupdate() {
        BasicCollection.prototype.activateAutoupdate.apply(this, arguments);
    },

    deActivateAutoupdate() {
        BasicCollection.prototype.deActivateAutoupdate.apply(this, arguments);
    },

    serializeData() {
        return this.map(client => {
            return client.serializeData();
        });
    },

    /**
     * @param {Object} updateOptions Set of options collection will be updated with.
     */
    update(updateOptions) {
        if (this.isUpdating) {
            this.isNeedUpdate = true;

            return false;
        }

        updateOptions = updateOptions || this.getPagingOptions();

        this.isUpdating = true;
        this.trigger('update:start');

        return this
            .fetch({
                data: _.extend(
                    this.getFetchParams(updateOptions),
                    this.getFilterParams().serializeAsQuery()
                )
            })
            .always(() => {
                this.isUpdating = false;

                if (this.isNeedUpdate) {
                    this.isNeedUpdate = false;
                    this.update();
                } else {
                    app.reqres.request('AUTOUPDATE_SKIP', false);
                    this.trigger('update:complete');
                }
            })
            .fail(xhr => {
                error.fromXHR(xhr);
            });
    },

    getPagingOptions() {
        return {
            limit: 100500
        };
    },

    mapFilterQueryOptions(options) {
        this.options.filterParams = new ClientFilterModel(options);
    },

    getFilterParams() {
        return this.options.filterParams;
    }
});

module.exports = ClientsCollection;
