const app = require('./../app');
const BasicCollection = require('./BasicCollection');
const ResourceFilterModel = require('../model/filter/ResourceFilterModel');

/**
 * @class TaskResourcesCollection
 * @extends BasicCollection
 */
const TaskResourcesCollection = BasicCollection.extend({
    url: '/api/v1.0/resource',
    model: require('../model/TaskResourceModel'),

    options: _.extend({}, BasicCollection.prototype.options),

    initialize(models, options) {
        this.mapQueryOptions(options);

        this.listenTo(this, 'reset', this.dropLastSelected);
        this.listenTo(this, 'change:selected', this.onSelectedChange);
        this.listenTo(this, 'change:filterparams:preset', this.setFilterParams);

        this.setFilterParams = this.setFilterParams.bind(this);

        app.stopListening(app, 'change:filterparams:preset');
        app.listenTo(app, 'change:filterparams:preset', this.setFilterParams);
    },

    update(updateOptions) {
        updateOptions = updateOptions || this.getPagingOptions();
        this.trigger('update:start');

        const data = _.extend(this.getFetchParams(updateOptions), this.getFilterParams().serializeAsQuery());

        return this.fetch({
            data,
            cancellable: !data.id
        }).then(() => {
            app.reqres.request('AUTOUPDATE_SKIP', false);
        });
    },

    fetchExistingResources() {
        return this.setFilterParams(new ResourceFilterModel(this.getCurrentItemsKeys()));
    },

    /**
     * @returns {ResourceFilterModel}
     */
    getFilterParams() {
        return this.options.filterParams;
    },

    mapFilterQueryOptions(options) {
        this.options.filterParams = new ResourceFilterModel(options);
    },

    batchAction(action, resources) {
        resources = (resources instanceof Array ? resources : this.where({ selected: true }));

        return TaskResourcesCollection.batchAction(action, resources);
    }

}, {
    batchAction(action, resources, options) {
        return BasicCollection.batchAction(action, resources, _.extend({
            url: ('/api/v1.0/batch/resources/' + action),
            actions: TaskResourcesCollection.BATCH_OPERATIONS
        }, options));
    },

    BATCH_OPERATIONS: {
        IMPORTANT: 'do_not_remove',
        RESTORE: 'restore',
        RESHARE: 'reshare',
        DELETE: 'delete',
        BACKUP: 'backup'
    }
});

module.exports = TaskResourcesCollection;
