import { difference } from 'lodash';

// eslint-disable-next-line no-unused-vars
const error = require('../error');
const app = require('../app');
const cookie = require('bemmy/src/blocks/component/cookie');
const TaskModel = require('../model/TaskModel');
const BasicCollection = require('./BasicCollection');
// eslint-disable-next-line no-unused-vars
const DateTimeIntervalModel = require('./../model/DateTimeIntervalModel');

/**
 * @class TasksCollection
 * @extends BasicCollection
 * @extends Backbone.Events
 */
const TasksCollection = BasicCollection.extend({

    url: '/api/v1.0/task',
    model: TaskModel,

    filterModelClass: require('../model/filter/FilterModel'),

    metricKey: 'TASKS-LOADED',

    initialize(models, options) {
        options = this.setDefaultFilter(options);

        this.options = _.extend({}, this.options, options);

        this.mapQueryOptions(options);

        this.setFilterParams = this.setFilterParams.bind(this);

        this.listenTo(this, 'reset', this.dropLastSelected);
        this.listenTo(this, 'change:selected', this.onSelectedChange);

        app.stopListening(app, 'change:filterparams:preset');
        app.listenTo(app, 'change:filterparams:preset', this.setFilterParams.bind(this));
    },

    setDefaultFilter(options) {
        if (this.isFilterParamsEmpty(options) && options.forPage === 'tasks') {
            options.author = cookie.getYandexLogin() || 'guest';
            options.created = '7_days';
        }

        return options;
    },

    isFilterParamsEmpty(options) {
        return Boolean(options && !difference(Object.keys(options), ['forPage', 'tags']).length);
    },

    /**
     * @returns {FilterModel}
     */
    getFilterParams() {
        return this.options.filterParams;
    },

    /**
     * @param {Object} updateOptions Set of options collection will be updated with.
     *
     * @return {JQueryDeferred}
     */
    update(updateOptions) {
        updateOptions = updateOptions || this.getPagingOptions();

        this.trigger('update:start');

        return this.fetch({
            data: _.extend(this.getFetchParams(updateOptions), this.getFilterParams().serializeAsQuery()),
            cancellable: true
        }).done(() => {
            app.reqres.request('AUTOUPDATE_SKIP', false);
        });
    },

    batchAction(action, tasks) {
        tasks = (tasks instanceof Array ? tasks : this.where({ selected: true }));

        return TasksCollection.batchAction(action, tasks);
    },

    mapFilterQueryOptions(options) {
        if (Object.keys(options).length === 0) {
            options.author = cookie.getYandexLogin();
        }

        const FilterModelClass = this.filterModelClass;

        this.options.filterParams = new FilterModelClass(FilterModelClass.parseOuterFilterValues(options));
    },

    fetchChildren() {
        const self = this;
        const parentIds = this.getParentIds();

        if (parentIds.length) {
            const childLevel = this.get(parentIds[0]).getChildLevel();

            return $.ajax({
                url: '/api/v1.0/task/?' +
                    'limit=100500&' +
                    'order=-id&' +
                    'children=true&' +
                    'hidden=true&' +
                    'parent=' + parentIds.join(',')
            }).done(data => {
                parentIds.forEach(parentId => {
                    const index = self.indexOf(self.get(parentId));

                    self.add(
                        data.items
                            .filter(childTaskData => {
                                return childTaskData.parent.id === parentId;
                            })
                            .map(childTaskData => {
                                const childTask = new TaskModel(childTaskData, { parse: true });

                                childTask.setChildLevel(childLevel + 1);

                                return childTask;
                            }),
                        { at: index + 1, silent: true }
                    );
                });
            })
                .then(() => {
                    return self.fetchChildren();
                });
        }

        return $.Deferred().resolve(this); // eslint-disable-line
    },

    getParentIds() {
        return this.filter(task => {
            return (task.getChildrenQnt() > 0 && !task.attributes.children.fetched);
        }).map(task => {
            task.attributes.children.fetched = true;

            return task.attributes.id;
        });
    }

}, {

    /**
     * @param {String}      action
     * @param {TaskModel[]} tasks
     *
     * @returns {JQueryDeferred}
     */
    batchAction(action, tasks, options) {
        return BasicCollection.batchAction(action, tasks, _.extend({
            url: ('/api/v1.0/batch/tasks/' + action),
            actions: TasksCollection.BATCH_OPERATIONS
        }, options));
    }
});

TasksCollection.BATCH_OPERATIONS = {
    STOP: 'stop',
    DELETE: 'delete',
    RESTART: 'start',
    SUSPEND: 'suspend',
    RESUME: 'resume',
    START: 'start',
    INC_PRIOR: 'increase_priority'
};

module.exports = TasksCollection;
