const router = require('../../router');

/**
 * @class UserPresetsCollection
 * @extends Backbone.Collection
 */
const UserPresetsCollection = Backbone.Collection.extend({

    url: '/api/v1.0/user/current/preferences/task-list',
    model: require('../../model/filter/FilterModel'),

    comparator: 'order',

    initialize(models, options) {
        this.url = ((options && options.url) ? options.url : this.url);
        this.model = ((options && options.model) ? options.model : this.model);
        this.lsKey = ('__sb_presets__' + this.url);

        this.listenTo(this, 'change remove', _.debounce(this.onPresetListUpdated, 100).bind(this));
    },

    update() {
        return this.fetch();
    },

    parse(data) {
        if (data instanceof Array) {
            return data;
        }

        return Object.keys(data).map(key => data[key]);
    },

    onPresetListUpdated() {
        this.savePresetList();
    },

    savePresetList() {
        return $.ajax({
            url: this.url,
            type: 'PUT',
            data: JSON.stringify(this.toJSON()),
            dataType: 'json',
            contentType: 'application/json'
        });
    },

    addPreset(preset) {
        const lastItem = this.at(this.length - 1);

        const order = (lastItem && lastItem.get('order') + 1) || 1;

        preset.set({
            order,
            forPage: this.getCurrentPage()
        });

        if (preset.get('attrs')) {
            preset.set('attrs', JSON.stringify(preset.get('attrs')));
        }

        this.add(preset);
        this.onPresetListUpdated();
    },

    toJSON() {
        const result = {};

        this.forEach((preset, i) => {
            result[i] = _.extend({
                title: preset.get('title'),
                order: preset.get('order')
            }, preset.getFilterParams());
        });

        return result;
    },

    updateCounters() {
        const self = this;
        let filterItemName = 'task';

        if (this.getCurrentPage() === 'resources') {
            filterItemName = 'resource';
        }

        this.each(filterModel => {
            if (self.presetForCurrentPage(filterModel.get('forPage'))) {
                filterModel.filterItemName = filterItemName;
                filterModel.getFound();
            }
        });
    },

    getCurrentPage() {
        return router.getCurrentPathFragment().replace(/\//g, '');
    },

    presetForCurrentPage(page) {
        return this.getCurrentPage() === page;
    }
});

module.exports = UserPresetsCollection;
