import React, { Component } from 'react';
import PropTypes from 'prop-types';
import b_ from 'b_';
import { isObject } from 'lodash';

import { YInput, YPopup } from 'y-components';

class DataList extends Component {
    constructor(props) {
        /* eslint max-statements: [1, 11] */

        super(props);

        this._block = b_.with('datalist');
        this.state = {};

        this.handleChange = this.handleChange.bind(this);
        this.handleFocus = this.handleFocus.bind(this);
        this.handleBlur = this.handleBlur.bind(this);
        this.handleKeyDown = this.handleKeyDown.bind(this);
        this.handleDelete = this.handleDelete.bind(this);
        this.handleToggle = this.handleToggle.bind(this);

        this.getSelected = this.getSelected.bind(this);

        this.refInputTarget = function (ref) {
            this._input = ref;
        }.bind(this);
    }

    componentDidMount() {
        this.setState({
            inputTarget: this._input
        });
    }

    handleSelect(value) {
        return e => {
            e.preventDefault();
            e.stopPropagation();

            this.props.onSelect(value);

            return false;
        };
    }

    handleChange(e) {
        this.props.onChange(e);
    }

    handleFocus() {
        this.props.onOpen();
        this.setState({
            focus: true
        });
    }

    handleBlur() {
        this.props.onClose();
        setTimeout(() => {
            this.setState({
                focus: false
            });
        }, 150);
    }

    handleKeyDown(e) {
        this.props.onKeyDown(e);
    }

    handleDelete(i) {
        return () => this.props.onDelete(i);
    }

    getSelected(item, k) {
        return (
            <div key={k} className={this._block('selected')}>
                <div className={this._block('selected-val')}>{item}</div>
                <div className={this._block('delete')} onClick={this.handleDelete(k)}>
                    <i className={b_('icon', { cross: 'mid' })}/>
                </div>
            </div>
        );
    }

    getData(data, active) {
        let lastTitle = '';

        return data.map((item, i) => {
            const title = lastTitle === item.title ? '' : item.title;

            lastTitle = item.title;

            const desc = isObject(item) ? item.desc : item;
            const isActive = active === i;
            const classes = this._block('popup-item', {
                active: isActive,
                disabled: item.disabled,
                title: Boolean(title)
            });

            return (
                <div
                    key={i}
                    data-title={title}
                    className={classes}
                    onClick={!item.disabled && this.handleSelect(item)}
                    >
                    {desc}
                </div>
            );
        });
    }

    handleToggle(state) {
        return () => {
            this.props.onToggle(state);
        };
    }

    render() {
        const {
            value,
            togglerValue,
            data,
            selected,
            visible,
            name,
            pin,
            size,
            placeholder,
            inPlace,
            mix
        } = this.props;
        const isVisible = this.state.inputTarget && this.state.focus && data.length && value && visible;
        const active = this.props.active || 0;

        return (
            <div className={`${mix} ${this._block({ pin, 'in-place': inPlace })}`}>
                {Array.isArray(selected) && selected.map(this.getSelected)}
                <div className={this._block('control')} ref={this.refInputTarget}>
                    <YInput
                        pin={pin}
                        size={size}
                        name={name}
                        value={value}
                        placeholder={placeholder}
                        onChange={this.handleChange}
                        onFocus={this.handleFocus}
                        onBlur={this.handleBlur}
                        onKeyDown={this.handleKeyDown}
                        autoComplete="off"
                        />
                    {inPlace ? (isVisible ? this.getData(data, active) : null) : (
                        <YPopup
                            mainOffset={8}
                            directions={['bottom-left', 'top-left']}
                            mix={`${mix} ${this._block('popup')}`}
                            size={'m'}
                            target={this.state.inputTarget}
                            visible={isVisible}
                            >
                            {this.getData(data, active)}
                        </YPopup>
                    )}
                </div>
                {
                    togglerValue !== undefined && <div className={this._block('toggler')}>
                        <span
                            className={this._block('toggler-item', { active: togglerValue === false })}
                            onClick={this.handleToggle(false)}
                            >
                            OR
                        </span>
                        <span
                            className={this._block('toggler-item', { active: togglerValue === true })}
                            onClick={this.handleToggle(true)}
                            >
                            AND
                        </span>
                    </div>
                }
            </div>
        );
    }
}

DataList.propTypes = {
    onChange: PropTypes.func,
    onToggle: PropTypes.func,
    onDelete: PropTypes.func,
    onSelect: PropTypes.func,
    onKeyDown: PropTypes.func,
    onOpen: PropTypes.func,
    onClose: PropTypes.func,
    data: PropTypes.array,
    selected: PropTypes.array,
    value: PropTypes.string,
    togglerValue: PropTypes.bool,
    name: PropTypes.string,
    pin: PropTypes.string,
    mix: PropTypes.string,
    size: PropTypes.string,
    placeholder: PropTypes.string,
    active: PropTypes.oneOfType([
        PropTypes.bool,
        PropTypes.number
    ]),
    visible: PropTypes.bool,
    inPlace: PropTypes.bool
};

DataList.defaultProps = {
    onChange: () => console.warn('DataList: onChange is not defined'), // eslint-disable-line
    onDelete: () => console.warn('DataList: onDelete is not defined'), // eslint-disable-line
    onSelect: () => console.warn('DataList: onSelect is not defined'), // eslint-disable-line
    onKeyDown: () => console.warn('DataList: onKeyDown is not defined'), // eslint-disable-line
    onOpen: () => console.warn('DataList: onOpen is not defined'), // eslint-disable-line
    onClose: () => console.warn('DataList: onClose is not defined'), // eslint-disable-line
    onToggle: () => {},
    data: [],
    selected: [],
    visible: false,
    active: 0
};

module.exports = DataList;
