import React, { Component } from 'react';
import PropTypes from 'prop-types';

import DataList from '../DataList';

class DataListContainer extends Component {
    constructor(props) {
        super(props);

        this.state = {
            active: null
        };

        this.handleOpen = this.handleOpen.bind(this);
        this.handleClose = this.handleClose.bind(this);
        this.handleSelect = this.handleSelect.bind(this);
        this.handleDelete = this.handleDelete.bind(this);
        this.handleChange = this.handleChange.bind(this);
        this.handleKeyDown = this.handleKeyDown.bind(this);
        this.handleToggle = this.handleToggle.bind(this);
    }

    handleOpen() {
        this.setState({
            visible: true
        });
    }

    handleClose() {
        setTimeout(() => {
            this.setState({
                visible: false
            });
        }, 200);
    }

    handleSelect(data) {
        const { value } = data;
        const { name, multi } = this.props;

        if (multi) {
            const { selected } = this.props;

            if (selected.indexOf(value) === -1) {
                selected.push(value);
                this.props.onSelectChange(selected);
                this.handleChange({ target: { name, value: '' }, type: 'select' });
            }
        } else {
            this.handleChange({ target: { name, value }, type: 'select' });
        }

        this.handleClose();
    }

    handleDelete(i) {
        const selected = this.props.selected.slice();

        selected.splice(i, 1);
        this.props.onSelectChange(selected);
    }

    handleChange(e) {
        const { key } = e;

        if (key !== ',') {
            this.props.onChange(e);
        }
    }

    handleKeyDown(e) {
        /* eslint complexity: [1, 8] */

        const { key } = e;

        switch (key) {
            case 'Tab':
                e.preventDefault();
                this.autoComplete();
                break;
            case 'ArrowDown':
                this.setActive('down');
                break;
            case 'ArrowUp':
                this.setActive('up');
                break;
            case 'Enter':
            case ',':
                this.selectActive();
                break;
            case 'Backspace':
                this.deleteLast();
                break;
            default:
                break;
        }
    }

    handleToggle(state) {
        if (state !== this.props.togglerValue) {
            this.props.onTogglerChange(state);
        }
    }

    autoComplete() {
        const { data, name } = this.props;
        const { active } = this.state;

        if (data[active]) {
            const value = data[active].value;

            this.setState({ active: 0 });
            this.handleChange({ target: { name, value }});
        }
    }

    setActive(dir) {
        const { length } = this.props.data;
        let { active } = this.state;

        if (dir === 'down') {
            active = active === null ? 0 : active + 1;
        } else {
            active = active === null ? length - 1 : active - 1;
        }

        active = ((active % length) + length) % length;

        this.setState({ active });
    }

    selectActive() {
        const { data, value } = this.props;
        const { active } = this.state;

        if (data[active]) {
            this.handleSelect(data[active]);
        } else if (value) {
            this.handleSelect({ value });
        }
    }

    deleteLast() {
        const { value, multi } = this.props;

        if (multi && !value.trim()) {
            const i = this.props.selected.length - 1;

            this.handleDelete(i);
        }
    }

    componentWillReceiveProps(newProps) {
        if (this.props.value !== newProps.value) {
            this.setState({
                visible: true,
                active: 0
            });
        }
    }

    render() {
        const { visible, active } = this.state;
        const { selected, data, value, name, placeholder, mix, togglerValue } = this.props;

        return React.createElement(DataList, {
            name,
            mix,
            placeholder,
            value,
            togglerValue,
            data,
            selected,
            visible,
            active,
            onSelect: this.handleSelect,
            onChange: this.handleChange,
            onDelete: this.handleDelete,
            onOpen: this.handleOpen,
            onClose: this.handleClose,
            onKeyDown: this.handleKeyDown,
            onToggle: this.handleToggle
        });
    }
}

DataListContainer.propTypes = {
    onChange: PropTypes.func,
    onSelectChange: PropTypes.func,
    onTogglerChange: PropTypes.func,
    selected: PropTypes.array,
    data: PropTypes.array,
    name: PropTypes.string,
    value: PropTypes.string,
    togglerValue: PropTypes.bool,
    multi: PropTypes.bool,
    placeholder: PropTypes.string,
    mix: PropTypes.string
};

DataListContainer.defaultProps = {
    data: []
};

module.exports = DataListContainer;
