import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import b_ from 'b_';

import { get } from 'lodash';

import YPopup from 'y-components/dist/y-popup';
import YInput from 'y-components/dist/y-input';
import YButton from 'y-components/dist/y-button';

import i18n from '../i18n';

class Extensible extends PureComponent {
    constructor(props) {
        /* eslint max-statements: [1, 11] */

        super(props);

        this._block = b_.with('extensible');
        this._texts = get(i18n, ['common']);

        this.state = {
            value: '',
            valid: true
        };

        this.refPopupTarget = ref => {
            this._popup = ref;
        };

        this.refInputTarget = ref => {
            this._input = ref;
        };

        this.handleShowPopup = this.handleShowPopup.bind(this);
        this.handleHidePopup = this.handleHidePopup.bind(this);
        this.handleSubmit = this.handleSubmit.bind(this);
        this.handleChange = this.handleChange.bind(this);
        this.handleKeyUp = this.handleKeyUp.bind(this);
    }

    componentDidMount() {
        this.setState({
            popupTarget: this._popup
        });
    }

    componentDidUpdate(prevProps, prevState) {
        if (this.state.popupVisible && !prevState.popupVisible) {
            this.handleInputFocus();
        }
    }

    handleInputFocus() {
        if (this._input) {
            setTimeout(() => this._input.control.focus(), 0);
        }
    }

    handleShowPopup() {
        this.setState({
            value: '',
            valid: true,
            popupVisible: true
        });
    }

    handleHidePopup() {
        this.setState({
            popupVisible: false
        });
    }

    handleSubmit() {
        const { value } = this.state;

        if (this.getValidation(value)) {
            this.handleHidePopup();
            this.props.onAdd({
                value
            });
        } else {
            this.setState({
                valid: false
            });
        }
    }

    handleChange(e) {
        this.setState({
            value: e.target.value,
            valid: true
        });
    }

    handleKeyUp(e) {
        switch (e.keyCode) {
        // Enter keyCode = 13
            case 13:
                this.handleSubmit();
                break;
                // Escape keyCode = 27
            case 27:
                this.handleHidePopup();
                break;
        // No default
        }
    }

    getValidation(value) {
        return value && (!this.props.pattern || new RegExp(this.props.pattern).test(value));
    }

    render() {
        const { children, mix, title, placeholder, extensible } = this.props;
        const { value, valid, popupTarget, popupVisible } = this.state;

        return (
            <div className={`${mix} ${this._block()}`}>
                <div
                    ref={this.refPopupTarget}
                    className={this._block('handler', { extensible })}
                    onClick={this.handleShowPopup}
                    >
                    {title}
                </div>
                <div className={this._block('content')}>
                    {children}
                </div>
                {
                    extensible ?
                        <YPopup
                            mix={this._block('popup')}
                            directions={['bottom-left', 'bottom-right']}
                            visible={popupVisible}
                            target={popupTarget}
                            mainOffset={0}
                            onShouldClose={this.handleHidePopup}
                            >
                            <YInput
                                mix={this._block('popup-input', { invalid: !valid })}
                                ref={this.refInputTarget}
                                type="text"
                                size="m"
                                value={value}
                                placeholder={placeholder || title || ''}
                                onChange={this.handleChange}
                                onKeyUp={this.handleKeyUp}
                                />
                            <YButton size="s" theme="action" onClick={this.handleSubmit}>
                                {this._texts.submit}
                            </YButton>
                            <YButton size="s" onClick={this.handleHidePopup}>
                                {this._texts.cancel}
                            </YButton>
                        </YPopup> :
                        null
                }
            </div>
        );
    }
}

Extensible.propTypes = {
    children: PropTypes.any,
    mix: PropTypes.string,
    title: PropTypes.string,
    placeholder: PropTypes.string,
    pattern: PropTypes.string,
    extensible: PropTypes.bool,
    onAdd: PropTypes.func
};

Extensible.defaultProps = {
    mix: '',
    extensible: true,
    onAdd: () => {}
};

module.exports = Extensible;
