/* eslint react/forbid-component-props: 0 */

import React, { Component } from 'react';
import PropTypes from 'prop-types';
import styled from 'styled-components';

import { SuggestUserMulti } from '../../../../components/SuggestUser';
import { SuggestAbcMulti } from '../../../../components/SuggestABCGroup';
import { SuggestRbMulti } from '../../../../components/SuggestRBGroup';
import { SuggestStaffGroupMulti } from '../../../../components/SuggestStaffGroup';

import { YInput, YSelect, YOption } from 'y-components';

const Wrap = styled.div`
    width: 100%;
    max-width: 440px;
    display: flex;
    flex-direction: column;
`;

const Row = styled.div`
    width: 100%;
    max-width: 440px;
    display: flex;
    flex-direction: row;
    margin-bottom: 5px;

    .y-input, .datalist {
        margin-right: 10px;
    }

    .datalist__selected {
        height: inherit !important;
    }
`;

const Icon = styled.span`
    line-height: 28px;
    display: inline-block;
    width: 30px;
    font-size: 13px;
    text-align: center;
    margin: 0 3px;
    cursor: pointer;
`;

class MembersSelector extends Component {
    constructor(props, context) {
        super(props, context);

        this.state = {
            sources: props.sources && props.sources.length ? props.sources : [{
                source: 'USER',
                group: ''
            }]
        };

        this.handleAddSource = this.handleAddSource.bind(this);
        this.handleDeleteSource = this.handleDeleteSource.bind(this);
        this.handleKeyChange = this.handleKeyChange.bind(this);

        this.renderSourceSelector = this.renderSourceSelector.bind(this);
        this.renderGroupSelector = this.renderGroupSelector.bind(this);
    }

    handleAddSource() {
        this.setState({
            sources: [
                ...this.state.sources,
                {
                    source: 'USER',
                    group: ''
                }
            ]
        });
    }

    handleDeleteSource(i) {
        return () => {
            const oldSources = this.state.sources || [];
            const newSources = [...oldSources.slice(0, i), ...oldSources.slice(i + 1)];
            const state = {
                sources: newSources
            };

            this.setState(state);
            this.props.onChange(state);
        };
    }

    handleKeyChange(i, key) {
        return e => {
            const value = e.target ? e.target.value : e;

            const oldSources = this.state.sources || [];
            const newSources = [
                ...oldSources.slice(0, i),
                Object.assign(oldSources[i], { [key]: value }),
                ...oldSources.slice(i + 1)
            ];
            const state = {
                sources: newSources
            };

            this.setState(state);
            this.props.onChange(state);
        };
    }

    renderSourceSelector(source, i) {
        return (
            <YSelect size="s" theme="normal" onChange={this.handleKeyChange(i, 'source')}>
                <YOption selected={source.source === 'USER'} value="USER">USER</YOption>
                <YOption selected={source.source === 'ABC'} value="ABC">ABC</YOption>
                <YOption selected={source.source === 'RB'} value="RB">RB</YOption>
                <YOption selected={source.source === 'STAFF'} value="STAFF">STAFF</YOption>
            </YSelect>
        );
    }

    renderGroupSelector(source, i) {
        const handleChange = this.handleKeyChange(i, 'group');
        const value =
            source.group instanceof Array ?
                source.group :
                source.group ?
                    source.group.split(',') :
                    [];

        switch (source.source) {
            case 'USER':
                return (
                    <SuggestUserMulti
                        placeholder={'Select user(s)'}
                        size="s"
                        onChange={handleChange}
                        value={value}
                        />
                );
            case 'ABC':
                return (
                    <SuggestAbcMulti
                        placeholder={'Select ABC group(s)'}
                        size="s"
                        onChange={handleChange}
                        value={value}
                        />
                );
            case 'RB':
                return (
                    <SuggestRbMulti
                        placeholder={'Select RB group(s)'}
                        size="s"
                        onChange={handleChange}
                        value={value}
                        />
                );
            case 'STAFF':
                return (
                    <SuggestStaffGroupMulti
                        placeholder={'Select Staff group(s)'}
                        size="s"
                        onChange={handleChange}
                        value={value}
                        />
                );
            default:
                return (
                    <YInput
                        placeholder={'Select item'}
                        name="sync"
                        size="s"
                        onChange={handleChange}
                        value={value}
                        />
                );
        }
    }

    render() {
        const { sources } = this.state;

        return (
            <Wrap>
                { sources.map((source, i) => {
                    return (
                        <Row key={i}>
                            {this.renderSourceSelector(source, i)}
                            {this.renderGroupSelector(source, i)}
                            <Icon className="fa fa-trash" onClick={this.handleDeleteSource(i)}/>
                            {
                                i === sources.length - 1 ?
                                    <Icon className="fa fa-plus" onClick={this.handleAddSource}/> :
                                    <Icon/>
                            }
                        </Row>
                    );
                })}
            </Wrap>
        );
    }
}

MembersSelector.propTypes = {
    sources: PropTypes.array,
    onChange: PropTypes.func
};

MembersSelector.defaultProps = {
    sources: [],
    onChange: () => {}
};

export default MembersSelector;
