/* eslint complexity: [1, 22] */

import React, { Component } from 'react';
import { connect } from 'react-redux';
import PropTypes from 'prop-types';
import styled from 'styled-components';
import { get, find } from 'lodash';
import axios from 'axios';

import { YInput } from 'y-components';

import MembersSelector from './MembersSelector';

import Modal from '../Modal';
import { SuggestAbc } from '../../../components/SuggestABCGroup';

import i18n from '../../i18n';

import API from '../../../api';

const Label = styled.div`
    display: inline-block;
    max-width: 160px;
    width: 219px;
`;

const Row = styled.div`
    display: flex;
    margin-bottom: 16px;
`;

const Content = styled.div`
    width: 100%;
    max-width: 440px;
    display: flex;

    .y-select {
        width: 115px !important;
        margin-right: 7px;
    }

    .datalist {
        width: 100%;
    }

    .datalist__selected {
        color: #000 !important;
        border: 0;
        text-transform: none;
        font-size: 13px;
        font-weight: normal;
        height: 23px;
        letter-spacing: 0;

        .datalist__selected-val::first-letter {
            color: red;
        }
    }
`;

const Mailtail = styled.span`
    white-space: nowrap;
    margin: 0 0 0 10px;
    color: #ccc;
`;

class Edit extends Component {
    constructor(props) {
        super(props);

        this.state = this.defaultState(props);

        this.handlePush = this.handlePush.bind(this);

        this.handleChange = this.handleChange.bind(this);
        this.handleMembersChange = this.handleMembersChange.bind(this);
        this.handleSaveClick = this.handleSaveClick.bind(this);
        this.handleCancelClick = this.handleCancelClick.bind(this);
        this.handleSynAbcChange = this.handleSynAbcChange.bind(this);

        this.getNewContent = this.getNewContent.bind(this);
        this.getUpdateContent = this.getUpdateContent.bind(this);
        this.getErrors = this.getErrors.bind(this);
    }

    defaultState({ data }) {
        return {
            name: data.name || '',
            email: data.email || '',
            sources: data.sources || {
                source: 'USER',
                group: []
            },
            abc: data.abc || '',
            users: data.users || data.members || []
        };
    }

    componentWillReceiveProps(newProps) {
        this.setState(
            this.defaultState(newProps),
            this.forceUpdate
        );
    }

    handlePush(payload) {
        const { data } = this.props;

        this.props.onPush(payload);

        if (data.name) {
            this.props.update(payload);
        } else {
            this.props.push(payload);
        }
    }

    handleChange(e) {
        if (e.target.value !== this.state[e.target.name]) {
            this.setState({ [e.target.name]: e.target.value });
        }
    }

    handleSynAbcChange(e) {
        this.setState({
            abc: e.target.value
        });
    }

    handleSaveClick() {
        return new Promise(resolve => this.getErrors().then(errors => {
            if (errors.length) {
                this.setState({
                    isShowErrors: true
                });

                return resolve(true);
            }

            const { name, email, sources, abc } = this.state;

            const result = {
                name,
                email,
                abc,
                sources: (sources ? sources instanceof Array ? sources : [sources] : []).map(item => {
                    if (item.group instanceof Array) {
                        item.group = item.group.join(',');
                    }

                    return item;
                })
            };

            this.handlePush(result);

            return resolve(false);
        }));
    }

    handleCancelClick() {
        this.errors = [];
        this.setState(this.defaultState(this.props));
    }

    handleMembersChange(state) {
        this.setState(state);
    }

    getErrors() {
        return new Promise((resolve, _) => {
            const { name, sources, email } = this.state;
            const errors = [];

            if (!name) {
                errors.push('You should specify name of the group.');
            }

            if (email) {
                if (email.indexOf('@') !== -1 || email.indexOf('yandex-team') !== -1) {
                    errors.push('You should specify email without @ and yandex-team postfix.');
                }
            }

            for (let i = 0; i < sources.length; i++) {
                const source = sources[i];

                if (source && (!source.group || !source.source)) {
                    errors.push('You should specify list of users or sync source.');
                    break;
                }

                if (
                    source &&
                    source.group &&
                    (source.group.indexOf('https://') !== -1 ||
                    source.group.indexOf('http://') !== -1 ||
                    /[а-яёА-ЯЁ].*/.test(source.group))
                ) {
                    errors.push('You should specify source name like a short target name in latin, not link or simple text.');
                }
            }

            const opts = {
                withCredentials: true
            };
            const promises = [];
            const items = [];

            for (let i = 0; i < sources.length; i++) {
                const source = sources[i];

                if (source && source.group && source.source) {
                    (source.group instanceof Array ? source.group : source.group.split(',')).map(item => item.trim()).forEach(item => {
                        if (source.source === 'USER') {
                            items.push(`User with login ${item} does not exist.`);

                            promises.push(axios
                                .get(`//api.staff.yandex-team.ru/v3/persons?_query=login==regex(%27^${item}$%27)`, opts)
                                .then(response => response.data.total));
                        }
                        if (source.source === 'RB') {
                            items.push(`RB group ${item} does not exist.`);

                            promises.push(axios
                                .get(`//a.yandex-team.ru/api/group?name=${item}`, opts)
                                .then(response => response.data.length));
                        }
                        if (source.source === 'ABC') {
                            items.push(`ABC group ${item} does not exist.`);

                            promises.push(axios
                                .get(`//search.yandex-team.ru/suggest/?version=2&layers=groups&groups.per_page=20&groups.query=s_type:service&text=${item}`, opts)
                                .then(response => response.data.groups.result.length));
                        }
                        if (source.source === 'STAFF') {
                            items.push(`Staff group ${item} does not exist.`);

                            promises.push(axios
                                .get(`//search.yandex-team.ru/suggest/?version=2&layers=groups&groups.query=s_type:department&text=${item}`, opts)
                                .then(response => response.data.groups.result.length));
                        }
                    });
                }
            }

            const allPromises = [...promises];

            if (allPromises && allPromises.length) {
                Promise.all([...promises])
                    .then(data => data.forEach((value, i) => {
                        if (!value && items[i]) {
                            errors.push(items[i]);
                        }

                        this.errors = errors;

                        return resolve(errors);
                    }));
            } else {
                return resolve([]);
            }
        });
    }

    getNewContent() {
        return (<div>
            <Row>
                <Label>{i18n.labels.groups.name}</Label>
                <YInput
                    name="name"
                    size="s"
                    onChange={this.handleChange}
                    value={this.state.name}
                    />
            </Row>
            <Row>
                <Label>{i18n.labels.groups.email}</Label>
                <Content>
                    <YInput
                        name="email"
                        size="s"
                        onChange={this.handleChange}
                        value={this.state.email}
                        />
                    <Mailtail>@yandex-team.ru</Mailtail>
                </Content>
            </Row>
            <Row>
                <Label>{i18n.labels.groups.abc}</Label>
                <Content>
                    <SuggestAbc
                        value={this.state.abc}
                        name="abc"
                        onChange={this.handleSynAbcChange}
                        />
                </Content>
            </Row>
            <Row>
                <Label>{i18n.labels.groups.sources}</Label>
                <Content>
                    <MembersSelector
                        sources={this.state.sources}
                        onChange={this.handleMembersChange}
                        />
                </Content>
            </Row>
        </div>);
    }

    getUpdateContent() {
        return (<div>
            <Row>
                <Label>{i18n.labels.groups.name}</Label>
                <Content>{this.state.name}</Content>
            </Row>
            <Row>
                <Label>{i18n.labels.groups.email}</Label>
                <Content>
                    <YInput
                        name="email"
                        size="s"
                        onChange={this.handleChange}
                        value={this.state.email}
                        />
                    <Mailtail>@yandex-team.ru</Mailtail>
                </Content>
            </Row>
            <Row>
                <Label>{i18n.labels.groups.abc}</Label>
                <Content>
                    <SuggestAbc
                        value={this.state.abc}
                        name="abc"
                        onChange={this.handleSynAbcChange}
                        />
                </Content>
            </Row>
            <Row>
                <Label>{i18n.labels.groups.sources}</Label>
                <Content>
                    <MembersSelector
                        sources={this.state.sources}
                        onChange={this.handleMembersChange}
                        />
                </Content>
            </Row>
        </div>);
    }

    render() {
        const { isShowErrors } = this.state;
        const isEditMode = Boolean(this.props.data && this.props.data.name);

        return (
            <Modal
                edit={isEditMode}
                label={this.props.label}
                onSubmit={this.handleSaveClick}
                onCancel={this.handleCancelClick}
                >
                {
                    this.props.data.name ?
                        this.getUpdateContent() :
                        this.getNewContent()
                }
                {
                    isShowErrors && this.errors.map((error, i) => <div key={i} className="red">{error}</div>)
                }
            </Modal>
        );
    }
}

Edit.propTypes = {
    data: PropTypes.object,
    label: PropTypes.object,
    push: PropTypes.func,
    update: PropTypes.func,
    onPush: PropTypes.func
};

Edit.defaultProps = {
    label: null,
    onPush: () => {}
};

function mapStateToProps(state, ownProps) {
    const data = ownProps.name ? find(
        get(state, ['groups', 'items'], []),
        item => {
            return item.name === ownProps.name;
        }
    ) : {};

    return {
        data: Object.assign({}, data)
    };
}

function mapDispatchToProps(dispatch) {
    return {
        update: payload => dispatch(API.groups.updateGroup(payload)),
        push: payload => dispatch(API.groups.pushGroup(payload))
    };
}

module.exports = connect(mapStateToProps, mapDispatchToProps)(Edit);
