import React from 'react';
import PropTypes from 'prop-types';
import b_ from 'b_';

import { YInput } from 'y-components';

class HintsInput extends React.Component {
    constructor(props) {
        super(...arguments);

        this.state = {
            values: props.value || [],
            value: ''
        };

        this._block = b_.with('hints');

        this.handleDelete = this.handleDelete.bind(this);
        this.handleChange = this.handleChange.bind(this);
        this.handleKeyDown = this.handleKeyDown.bind(this);
        this.handleToggle = this.handleToggle.bind(this);
        this.handleBlur = this.handleBlur.bind(this);

        this.selectActive = this.selectActive.bind(this);
        this.deleteLast = this.deleteLast.bind(this);

        this.getSelected = this.getSelected.bind(this);
    }

    handleDelete(i) {
        return () => {
            this.setState({
                values: [].concat(
                    this.state.values.slice(0, i),
                    this.state.values.slice(i + 1)
                )
            }, () => {
                this.handleChange(this.state.values);
            });
        };
    }

    handleChange(values) {
        this.props.onChange(values);
    }

    handleKeyDown(e) {
        const { key } = e;

        switch (key) {
            case 'Tab':
            case 'Enter':
                e.preventDefault();
                this.selectActive(e);
                break;
            case 'Backspace':
                this.deleteLast(e);
                break;
            default:
                break;
        }
    }

    handleToggle(state) {
        return () => {
            this.props.onToggle(state);
        };
    }

    handleBlur(e) {
        this.selectActive(e);
    }

    selectActive(e) {
        if (e.target.value !== '') {
            this.setState({
                values: [].concat(this.state.values, [e.target.value]),
                value: ''
            }, () => {
                this.handleChange(this.state.values);
            });
        }
    }

    deleteLast(e) {
        if (!e.target.value) {
            this.setState({
                values: this.state.values.slice(0, -1),
                value: ''
            }, () => {
                this.handleChange(this.state.values);
            });
        }
    }

    getSelected(item, k) {
        return (
            <div key={k} className={this._block('selected')}>
                <div className={this._block('selected-val')}>{item}</div>
                <div className={this._block('delete')} onClick={this.handleDelete(k)}>
                    <i className={b_('icon', { cross: 'mid' })}/>
                </div>
            </div>
        );
    }

    render() {
        const {
            togglerValue,
            placeholder
        } = this.props;
        const { value, values } = this.state;

        return (
            <div className={this._block()}>
                {Array.isArray(values) && values.map(this.getSelected)}
                <div className={this._block('control')}>
                    <YInput
                        size={'m'}
                        value={value}
                        placeholder={placeholder}
                        onBlur={this.handleBlur}
                        onKeyDown={this.handleKeyDown}
                        autoComplete="off"
                        />
                </div>
                {
                    togglerValue !== undefined && <div className={this._block('toggler')}>
                        <span
                            className={this._block('toggler-item', { active: togglerValue === false })}
                            onClick={this.handleToggle(false)}
                            >
                            OR
                        </span>
                        <span
                            className={this._block('toggler-item', { active: togglerValue === true })}
                            onClick={this.handleToggle(true)}
                            >
                            AND
                        </span>
                    </div>
                }
            </div>
        );
    }
}

HintsInput.propTypes = {
    placeholder: PropTypes.string,
    togglerValue: PropTypes.bool,
    value: PropTypes.arrayOf(PropTypes.string),
    onToggle: PropTypes.func,
    onChange: PropTypes.func
};

HintsInput.defaultProps = {
    placeholder: '',
    onChange: () => {},
    onToggle: () => {}
};

module.exports = HintsInput;
