/* global WebSocket */

import React, { Component } from 'react';
import PropTypes from 'prop-types';

import { YSpin } from 'y-components';

class LogsTail extends Component {
    constructor(props) {
        super(props);

        this.state = {
            size: null,
            isLoading: true,
            error: false
        };

        this._refLogsBottom = this._refLogsBottom.bind(this);
        this._closeSocket = this._closeSocket.bind(this);
        this._closeSocketAndFlushData = this._closeSocketAndFlushData.bind(this);
        this._errorSocket = this._errorSocket.bind(this);
        this._getData = this._getData.bind(this);
        this._getUrl = this._getUrl.bind(this);
        this.scrollToBottom = this.scrollToBottom.bind(this);
        this.initSocket = this.initSocket.bind(this);
    }

    componentWillReceiveProps(nextProps) {
        if (this.props.type !== nextProps.type) {
            this.setState({ isLoading: true });
            this.initSocket(nextProps);
        }
    }

    componentDidMount() {
        this.initSocket();
    }

    componentWillUnmount() {
        this._closeSocket();
    }

    _refLogsBottom(el) {
        this.refLogsBottom = el;
    }

    _closeSocket() {
        if (this._socket) {
            clearInterval(this._counter);
            this._socket.onclose = function () {};
            this._socket.close();
        }
    }

    _closeSocketAndFlushData() {
        this._closeSocket();
        this.setState({ data: '', size: null });
    }

    _errorSocket() {
        this._closeSocketAndFlushData();
        this.setState({
            data: '',
            error: 'WebSocket has been closed.',
            isLoading: false
        });
    }

    scrollToBottom() {
        if (this.refLogsBottom && this.props.scroll) {
            this.refLogsBottom.scrollIntoView({ behavior: 'smooth' });
        }
    }

    initSocket(_props = this.props) {
        this._closeSocketAndFlushData();

        if (_props.type && window.WebSocket) {
            this._socket = new WebSocket(this._getUrl(_props.type));

            this._socket.onopen = () => {
                const params = {
                    size: this.state.size || null
                };

                this._socket.send(JSON.stringify(params));

                this._counter = setInterval(() => {
                    this._socket.send(JSON.stringify({
                        size: this.state.size || null
                    }));
                }, 2000);
            };

            this._socket.onmessage = message => {
                const result = JSON.parse(message.data);
                const { data, size } = result;

                if (data) {
                    this.setState({ data: this._getData(data) });
                    this.scrollToBottom();
                }

                if (this.state.isLoading || this.state.size !== size || this.state.error) {
                    this.setState({
                        isLoading: false,
                        size,
                        error: false
                    });
                }
            };

            this._socket.onclose = this._errorSocket;
            this._socket.onerror = this._errorSocket;
            window.onbeforeunload = this._closeSocketAndFlushData;
        }
    }

    _getData(data) {
        return ((this.state.data + data) || '').split('\n').splice(-3000).join('\n');
    }

    _getUrl(type) {
        const base = window.location.origin.indexOf('www-sandbox1.n') === -1 ?
            'wss://proxy.sandbox.yandex-team.ru' :
            'wss://proxy-sandbox1.n.yandex-team.ru';
        const url = base + '/tail/' + this.props.id + '/log1/' + type + '.log';

        return url;
    }

    render() {
        if (this.state.isLoading) {
            return (
                <div className="section__h">
                    <YSpin progress size="xxs"/>
                </div>
            );
        }

        return (
            <div>
                <pre className="section__h">
                    {this.state.error || this.state.data}
                </pre>
                <div ref={this._refLogsBottom}/>
            </div>

        );
    }
}

LogsTail.propTypes = {
    type: PropTypes.string,
    scroll: PropTypes.bool,
    id: PropTypes.number
};

LogsTail.defaultProps = {
    type: 'common'
};

module.exports = LogsTail;
