import React, { Component } from 'react';
import PropTypes from 'prop-types';

import b_ from 'b_';

import { YButtonGroup, YButton } from 'y-components';

class Pagination extends Component {
    constructor(props) {
        super(props);

        this.state = {
            limit: props.limit,
            current: props.current,
            hasNext: props.hasNext
        };

        this.handleChangeLimit = this.handleChangeLimit.bind(this);
        this.handleClick = this.handleClick.bind(this);
        this.getPage = this.getPage.bind(this);
    }

    handleChangeLimit(limit) {
        const self = this;

        return function () {
            self.setState({
                limit
            });

            self.props.onChange({ limit, page: self.state.current });
        };
    }

    handleClick(page) {
        const self = this;

        return function () {
            self.setState({
                current: page
            });

            self.props.onChange({ limit: self.state.limit, page });
        };
    }

    getGap() {
        return (
            <YButton key={'gap-1'} mix={b_('pagination', 'page')}>...</YButton>
        );
    }

    getPage(num, arrow) {
        /* eslint complexity: [1, 7] */
        if (num < 1 && !arrow) {
            return null;
        }

        const current = this.props.current;
        const arrows = {
            left: '←',
            right: '→'
        };
        const theme = arrow ? 'normal' : num === current ? 'radio' : 'normal';
        const Num = arrow ? arrows[arrow] : num;
        const key = arrow ? `${num}-${arrow}` : num;

        return (
            <YButton
                key={key}
                mix={b_('pagination', 'page')}
                theme={theme}
                value={num}
                disabled={num < 1}
                name="pagination"
                onClick={this.handleClick(num)}
                >
                {Num}
            </YButton>
        );
    }

    render() {
        /* eslint max-statements: [1, 25] */
        /* eslint complexity: [1, 10] */
        const current = this.props.current;
        const total = this.props.total;
        const limits = this.props.limits;
        const limit = this.props.limit;
        const hasNext = this.props.hasNext;
        const smallestLimit = (limits && limits.length && limits[0]) || 20;

        if (!hasNext && current === 1 && limit === smallestLimit) {
            return null;
        }

        const limitsArray = _.uniq(limits.concat([limit])).sort((a, b) => a > b);
        const Limits = limitsArray.map((lim, key) => (
            <YButton
                key={key}
                mix={b_('pagination', 'limit-item')}
                onClick={this.handleChangeLimit(lim)}
                theme={lim === this.state.limit ? 'radio' : 'normal'}
                value={lim}
                name="pagination-limit"
                >{lim}</YButton>
        ));
        const Pages = [];

        Pages.push(this.getPage(current - 1, 'left'));
        Pages.push(this.getPage(current - 2));
        Pages.push(this.getPage(current - 1));
        Pages.push(this.getPage(current));

        if (hasNext) {
            Pages.push(this.getPage(current + 1));
        }

        if (total) {
            const totalPages = Math.ceil(total / limit);

            Pages.push(this.getGap());
            Pages.push(this.getPage(totalPages - 1));
            Pages.push(this.getPage(totalPages));
        }

        if (hasNext) {
            Pages.push(this.getPage(current + 1, 'right'));
        }

        return (
            <div className="pagination">
                <div className={b_('pagination', 'pages')}>
                    <YButtonGroup values={[current]} size="s">
                        {Pages}
                    </YButtonGroup>
                </div>
                <div className={b_('pagination', 'limit')}>
                    <YButtonGroup values={[limit]} size="s">
                        {Limits}
                    </YButtonGroup>
                </div>
            </div>
        );
    }
}

Pagination.propTypes = {
    limit: PropTypes.number,
    limits: PropTypes.array,
    current: PropTypes.number,
    total: PropTypes.number,
    hasNext: PropTypes.bool
};

Pagination.defaultProps = {
    limit: 20,
    limits: [20, 50, 100],
    current: 5,
    hasNext: true,
    onChange() {}
};

module.exports = Pagination;
