import React, { Component } from 'react';
import PropTypes from 'prop-types';
import b_ from 'b_';

import { YInput, YPopup, YButton, YSelect, YOption } from 'y-components';

class ParameterInput extends Component {
    constructor(props) {
        super(props);

        this.refInputTarget = function (ref) {
            this._input = ref;
        }.bind(this);

        this._block = b_.with('parameter');

        this.state = {
            value: this._parseValue(),
            isEdit: false,
            oldKey: '',
            newKey: '',
            newValue: '',
            newValueType: 'string'
        };

        this.handleDelete = this.handleDelete.bind(this);
        this.handleToggle = this.handleToggle.bind(this);
        this.handleChangeKey = this.handleChangeKey.bind(this);
        this.handleChangeValue = this.handleChangeValue.bind(this);
        this.handleSave = this.handleSave.bind(this);
        this.handleChangeValueType = this.handleChangeValueType.bind(this);

        this._parseValue = this._parseValue.bind(this);
        this._compressValue = this._compressValue.bind(this);
    }

    _parseValue() {
        const { value } = this.props;

        const keys = Object.keys(value || {});

        return keys.reduce((result, key) => {
            result[key] = {
                key,
                value: value[key],
                type: typeof value[key]
            };

            return result;
        }, {});
    }

    _compressValue(value) {
        return Object.keys(value || {}).reduce((result, key) => {
            result[key] = value[key].value;

            return result;
        }, {});
    }

    componentDidMount() {
        this.setState({
            inputTarget: this._input
        });
    }

    handleEdit(key, value) {
        return e => {
            this.setState({
                oldKey: key,
                newKey: key,
                newValue: value,
                newValueType: this.state.value[key].type,
                isEdit: true
            });

            e.preventDefault();
            e.stopPropagation();

            return false;
        };
    }

    handleDelete(key) {
        return e => {
            const newState = Object.assign({}, this.state.value);

            delete newState[key];

            this.setState({
                value: newState
            });

            e.preventDefault();
            e.stopPropagation();

            this.props.onChange(this._compressValue(newState));
        };
    }

    handleToggle(e) {
        if (e.target.className !== 'y-select__text') {
            const { isEdit } = this.state;

            if (isEdit) {
                this.setState({
                    newKey: '',
                    newValue: '',
                    newValueType: 'string',
                    oldKey: ''
                });
            }

            this.setState({
                isEdit: !isEdit
            });
        }
    }

    handleTogglerChange(state) {
        return e => {
            this.props.onToggle(state);

            e.preventDefault();
            e.stopPropagation();

            return false;
        };
    }

    handleChangeKey(e) {
        this.setState({
            newKey: e.target.value
        });
    }

    handleChangeValue(e) {
        this.setState({
            newValue: e.target.value
        });
    }

    handleChangeValueType(e) {
        this.setState({
            newValueType: e.target.value
        });
    }

    handleSave() {
        /* eslint max-statements: [1, 15] */
        /* eslint complexity: [1, 16] */
        const { hasExtendedTypes } = this.props;
        const { newKey, newValue, oldKey, value, newValueType } = this.state;

        if (newKey === '' || (newValue === '' && newValueType !== 'null')) {
            return false;
        }

        const isEditMode = oldKey !== '';
        const newState = Object.assign({}, value);

        if (isEditMode) {
            delete newState[oldKey];
        }

        let castedValue = newValue;

        switch (newValueType) {
            case 'number':
                castedValue = Number.isNaN(Number(newValue)) ? 0 : Number(newValue);
                break;
            case 'boolean':
                castedValue = (newValue === 'false' || newValue === 'False' || newValue === '0') ?
                    false : (newValue === 'true' || newValue === 'True' || newValue === '1') ?
                        true : Boolean(newValue);
                break;
            default:
                castedValue = newValue;
        }

        if (hasExtendedTypes && newValueType === 'null') {
            castedValue = null;
        }

        newState[newKey] = {
            key: newKey,
            value: castedValue,
            type: newValueType
        };

        this.setState({
            newValue: '',
            newValueType: 'string',
            newKey: '',
            oldKey: '',
            value: newState,
            isEdit: false
        });

        this.props.onChange(this._compressValue(newState));
    }

    render() {
        const { togglerValue, placeholder, hasExtendedTypes } = this.props;
        const { value, inputTarget, isEdit, newKey, newValueType, newValue } = this.state;
        const keys = Object.keys(value);

        const self = this;

        return (
            <div>
                <div className={this._block()} ref={this.refInputTarget} onClick={this.handleToggle}>
                    {
                        keys.map((key, i) => {
                            return (
                                <div key={i} className={self._block('selected')}>
                                    <div
                                        className={self._block('selected-val')}
                                        onClick={self.handleEdit(key, value[key].value)}
                                        >
                                        {key}<span> = </span>{`${value[key].value}`}
                                    </div>
                                    <div className={self._block('delete')} onClick={self.handleDelete(key)}>
                                        <i className={b_('icon', { cross: 'mid' })}/>
                                    </div>
                                </div>
                            );
                        })
                    }
                    {
                        (!keys || !keys.length) && <div className={this._block('placeholder')}>{placeholder}</div>
                    }
                    <div className={this._block('control')}/>
                    {
                        togglerValue !== undefined && <div className={this._block('toggler')}>
                            <span
                                className={this._block('toggler-item', { active: togglerValue === true })}
                                onClick={this.handleTogglerChange(true)}
                                >
                                OR
                            </span>
                            <span
                                className={this._block('toggler-item', { active: togglerValue === false })}
                                onClick={this.handleTogglerChange(false)}
                                >
                                AND
                            </span>
                        </div>
                    }
                </div>
                <YPopup
                    mainOffset={8}
                    directions={['bottom-left', 'top-left']}
                    mix={this._block('popup')}
                    size={'m'}
                    target={inputTarget}
                    visible={isEdit}
                    onShouldClose={this.handleToggle}
                    >
                    <YInput
                        size="s"
                        theme="normal"
                        mix={self._block('key')}
                        placeholder={'Key'}
                        value={newKey}
                        onChange={this.handleChangeKey}
                        />
                    <YInput
                        size="s"
                        theme="normal"
                        mix={self._block('value')}
                        placeholder={'Value'}
                        value={newValue}
                        onChange={this.handleChangeValue}
                        />
                    <YSelect
                        disabled={newValue === '' && !hasExtendedTypes}
                        mix={this._block('type-selector')}
                        size="s"
                        theme="normal"
                        onChange={this.handleChangeValueType}
                        >
                        {
                            [
                                <YOption
                                    key="string"
                                    selected={newValueType === 'string'}
                                    value="string"
                                    >
                                        String
                                </YOption>,
                                <YOption
                                    key="number"
                                    selected={newValueType === 'number'}
                                    value="number"
                                    >
                                        Number
                                </YOption>,
                                <YOption
                                    key="boolean"
                                    selected={newValueType === 'boolean'}
                                    value="boolean"
                                    >
                                        Boolean
                                </YOption>,
                                hasExtendedTypes && <YOption
                                    key="null"
                                    selected={newValueType === 'null'}
                                    value="null"
                                    >
                                        Null
                                </YOption>
                            ].filter(Boolean)
                        }
                    </YSelect>
                    <YButton
                        size="s"
                        theme="action"
                        disabled={newKey === '' || newValue === ''}
                        onClick={this.handleSave}
                        >
                            Save
                    </YButton>
                </YPopup>
            </div>
        );
    }
}

ParameterInput.propTypes = {
    placeholder: PropTypes.string,
    togglerValue: PropTypes.bool,
    onChange: PropTypes.func,
    onToggle: PropTypes.func,
    value: PropTypes.object, // eslint-disable-line
    hasExtendedTypes: PropTypes.bool
};

ParameterInput.defaultProps = {
    placeholder: '',
    hasExtendedTypes: false,
    onChange: () => {},
    onToggle: () => {}
};

module.exports = ParameterInput;
