/* eslint camelcase: 0 */
/* eslint react/no-danger: 0 */
import React, { Component } from 'react';
import { connect } from 'react-redux';
import propTypes from 'prop-types';
import { get } from 'lodash';
import styled from 'styled-components';

import { YSpin, YButtonGroup, YButton } from 'y-components';
import UserLogin from '../../UserLogin';

const Table = styled.table`
    width: 100%;
    background: #fff;
    transition: opacity .3s ease;

    td:first-child, th:first-child {
        padding-left: 20px;
    }
`;

const HeadCell = styled.th`
    box-sizing: border-box;
    white-space: nowrap;
    vertical-align: middle;
    text-align: left;
    padding: 9px 5px;
    background: #eee;
`;

const Cell = styled.td`
    font-size: 12px;
    box-sizing: border-box;
    white-space: nowrap;
    min-height: 36px;
    vertical-align: top;
    box-shadow: #ebebeb 0 1px 0 inset;
    padding: 9px 5px 6px;
`;

import api from '../../../api/semaphores';

const TYPES = {
    RELATIVE_TO_CREATE: 2,
    RELATIVE_TO_PREVIOUS: 3,
    RELATIVE_TO_NOW: 4
};

class SemaphoreView extends Component {
    constructor() {
        super(...arguments);

        this.state = {
            time: this.getDefaultFormat()
        };

        this.handleTimeChange = this.handleTimeChange.bind(this);
        this.formatTime = this.formatTime.bind(this);
        this.computeData = this.computeData.bind(this);
    }

    componentDidMount() {
        if (!this.props.audit.length) {
            this.props.fetchSemaphoresAuditById(this.props.id);
        }
    }

    getDefaultFormat() {
        const state = (window.localStorage &&
            window.localStorage.getItem &&
            window.localStorage.getItem('SINGLETASK_HISTORY_FORMAT') !== '1' &&
            window.localStorage.getItem('SINGLETASK_HISTORY_FORMAT')) ||
            TYPES.RELATIVE_TO_CREATE;

        return [Number(state)];
    }

    handleTimeChange(e, change) {
        const values = this.state[change.name];
        const index = values.indexOf(change.value);

        if (index !== -1) {
            return;
        }

        this.setState({ time: [change.value] });

        if (window.localStorage && window.localStorage.setItem) {
            window.localStorage.setItem('SINGLETASK_HISTORY_FORMAT', change.value);
        }
    }

    formatTime(current, previous) {
        const _time = moment(current);
        const absolute = {
            main: _time.format('HH:mm:ss'),
            secondary: _time.format('DD.MM.YYYY')
        };

        switch (this.state.time[0]) {
            case TYPES.RELATIVE_TO_NOW:
                return {
                    absolute,
                    main: _time.fromNow(),
                    secondary: ''
                };
            default:
                if (_time.isSame(previous)) {
                    return {
                        absolute,
                        main: _time.fromNow(),
                        secondary: ''
                    };
                }

                return {
                    absolute,
                    main: moment.duration(_time.diff(previous)).humanize(),
                    secondary: '+'
                };
        }
    }

    computeData(data) {
        data.forEach(function (item, i) {
            if (item.time) {
                switch (this.state.time[0]) {
                    case TYPES.RELATIVE_TO_CREATE:
                        data[i]._time = this.formatTime(item.time, (data[0] || {}).time);
                        break;
                    case TYPES.RELATIVE_TO_PREVIOUS:
                        data[i]._time = this.formatTime(item.time, (data[i - 1] || item).time);
                        break;
                    default:
                        data[i]._time = this.formatTime(item.time);
                }
            }
        }, this);

        return data;
    }

    render() {
        /* eslint complexity: [1, 7] */

        const { audit, isAdmin } = this.props;
        const isLoading = !audit || audit.isLoading;
        const data = !isLoading && this.computeData(audit);
        const cols = isAdmin ?
            ['Title', 'Message', 'Author', 'Server', 'Remote IP'] :
            ['Title', 'Message', 'Author', 'Remote IP'];

        if (data && !data.length) {
            return <div className="section__i-clean">There is no history for semaphore.</div>;
        }

        return isLoading ?
            <div className="section__i-clean">
                <YSpin progress size="xxs"/>
            </div> :
            <div>
                <div className="section__i-clean">
                    <YButtonGroup theme="normal" size="s" values={this.state.time} onChange={this.handleTimeChange}>
                        <YButton name={'time'} value={TYPES.RELATIVE_TO_CREATE}>Relative to creation</YButton>
                        <YButton name={'time'} value={TYPES.RELATIVE_TO_PREVIOUS}>Relative to previous</YButton>
                        <YButton name={'time'} value={TYPES.RELATIVE_TO_NOW}>Relative to current time</YButton>
                    </YButtonGroup>
                </div>
                <Table>
                    <thead>
                        <tr>
                            {cols.map((col, i) => (
                                <HeadCell key={col + i}>
                                    <span>{col}</span>
                                </HeadCell>
                            ))}
                        </tr>
                    </thead>
                    <tbody>
                        {data.map((data, i) => {
                            return (
                                <tr key={i}>
                                    <Cell>
                                        <span className="gray">{data._time.absolute.secondary}</span> {data._time.absolute.main}{' '}
                                        <span className="t__time__nowrap">
                                            <span className="gray">({data._time.secondary}</span>{data._time.main}<span className="gray">)</span>
                                        </span>
                                    </Cell>
                                    <Cell>{data.description || '–'}</Cell>
                                    <Cell>{data.author ? <UserLogin login={data.author}/> : '–'}</Cell>
                                    { isAdmin && <Cell>{data.target || '–'}</Cell> }
                                    <Cell>{data.source || '–'}</Cell>
                                </tr>
                            );
                        })}
                    </tbody>
                </Table>
            </div>;
    }
}

SemaphoreView.propTypes = {
    id: propTypes.oneOfType([
        propTypes.number,
        propTypes.string
    ]),
    audit: propTypes.oneOfType([
        propTypes.arrayOf(propTypes.object),
        propTypes.object
    ]),
    fetchSemaphoresAuditById: propTypes.func,
    isAdmin: propTypes.bool
};

SemaphoreView.defaultProps = {
    semaphore: {},
    isAdmin: false
};

function mapStateToProps(state, props) {
    return {
        audit: get(state, ['semaphores', 'audit', String(props.id)], [])
    };
}

function mapDispatchToProps(dispatch) {
    return {
        fetchSemaphoresAuditById: id => dispatch(api.fetchSemaphoresAuditById(id))
    };
}

module.exports = connect(mapStateToProps, mapDispatchToProps)(SemaphoreView);
