/* eslint camelcase: 0 */
/* eslint react/no-danger: 0 */
/* eslint react/forbid-component-props: 0 */

import React, { Component } from 'react';
import { connect } from 'react-redux';
import propTypes from 'prop-types';
import { get } from 'lodash';
import styled from 'styled-components';

import clipboardClientMixin from '../../../views/components/clipboardClient/ClipboardClientMixin';
import Weather from '../../../views/helpers/getWeather';
import Timer from '../../../helpers/utils/Timer';

import { YSpin } from 'y-components';
import { SimpleTable } from 'unity-table';

import UserLogin from '../../UserLogin';
import Label from '../../common/Tags/Label';

const Wrap = styled.div`
    .unity-table {
        table-layout: fixed;

        .t__type-desc.t__type-desc_inline {
            max-width: 100%;
        }
    }
`;

import api from '../../../api/tasks';

const columns = [
    { name: 'id', sortable: true, width: '8.5%' },
    { name: 'status', sortable: true, width: '6.5%' },
    { name: 'type', sortable: true, width: '56%' },
    { name: 'acquired', width: '8.5%' },
    { name: 'updated', sortable: true, width: '13%' },
    { name: 'owner', sortable: true, width: '7.5%' }
];

class SemaphoreView extends Component {
    constructor() {
        super(...arguments);

        this.state = {
            sort: { field: 'id', asc: true },
            faded: false
        };

        this.mapData = this.mapData.bind(this);
        this.fetchTasks = this.fetchTasks.bind(this);
        this._buildIds = this._buildIds.bind(this);
        this.handleSortChange = this.handleSortChange.bind(this);
    }

    componentDidMount() {
        clipboardClientMixin.initializeClipboardClient(
            '.id-proxy-copier',
            'ID has been copied to clipboard.'
        );

        this.fetchTasks(this.props.semaphore.tasks);
    }

    componentWillReceiveProps(newProps) {
        if (JSON.stringify(this.props.semaphore) !== JSON.stringify(newProps.semaphore)) {
            this.setState({ faded: false });
        }

        if (
            newProps.semaphore.tasks &&
            newProps.semaphore.tasks.length && (
                !this.props.semaphore.tasks ||
                newProps.semaphore.tasks.length !== this.props.semaphore.tasks.length
            )
        ) {
            this.fetchTasks(newProps.semaphore.tasks);
        }
    }

    _buildIds(tasks) {
        tasks = tasks || this.props.semaphore.tasks;

        let ids = null;

        if (tasks && tasks.length) {
            ids = tasks.reduce((result, task) => {
                result.push(task.task_id);

                return result;
            }, []);
        }

        return ids;
    }

    fetchTasks(tasks) {
        const id = this._buildIds(tasks);

        if (id) {
            const { asc, field } = this.state;

            const params = {
                limit: 200,
                children: true,
                hidden: true,
                id: id.join(',')
            };

            if (field) {
                params.order = (asc ? '+' : '-') + field;
            }

            this.props.fetchTaskList(params);
        }
    }

    formatTime(t) {
        const time = Timer.serializeTimeMark(t);

        return <span>{`${time.date} ${time.time} `}(<span className="gray">{time.comment}</span>)</span>;
    }

    getWeight(data) {
        const { semaphore } = this.props;
        const item = semaphore.tasks.find(item => {
            return item.task_id === data.id;
        });

        return item ? item.weight : null;
    }

    handleSortChange(e) {
        this.setState({
            sort: {
                field: e.field,
                asc: e.asc
            },
            faded: true
        });
    }

    mapData(data) {
        return ({
            id: (
                <div>
                    <a className="t__console link id" href={'/task/' + data.id + '/view'}>{data.id}</a>
                    &nbsp;
                    <span className="link external fa fa-files-o id-proxy-copier" data-clipboard-text={data.id}/>
                </div>
            ),
            status: (data.status ?
                (<div>
                    <p className={`status status_${(data.status || '').toLowerCase()}`}>{data.status}</p>
                </div>) :
                <span>...</span>
            ),
            type: (data.type ? <div className="t__type-desc t__type-desc_inline">
                <span className="t__type t__type_weather">
                    <a
                        className={`t__type_icon t__type_weather t__type_weather_${Weather.getWeather(data.scores)}`}
                        title={`${data.scores}/10 last runs are OK`}
                        />
                    {data.type}
                </span>
                {
                    data.tags && <p className="t__desc t__gray">
                        {
                            data.tags.map(tag => {
                                return <Label tag={tag} key={tag}/>;
                            })
                        }
                    </p>
                }
                <p className="t__desc t__gray" dangerouslySetInnerHTML={{ __html: data.description }}/>
            </div> : null),
            acquired: this.getWeight(data),
            updated: data.time ? this.formatTime(data.time.updated) : null,
            owner: <UserLogin login={data.owner}/>
        });
    }

    getTableHead() {
        return {
            id: 'Task ID',
            status: 'Status',
            type: 'Type and description',
            acquired: 'Acquired weight',
            updated: 'Updated',
            owner: 'Owner'
        };
    }

    render() {
        /* eslint complexity: [1, 9] */
        const { semaphore, tasks } = this.props;
        const isLoading = !semaphore.tasks || tasks.isLoading;
        const data = !isLoading && tasks;

        if (!isLoading && (!data || (data && !data.length))) {
            return <div className="section__i-clean">There is no any tasks.</div>;
        }

        return isLoading ?
            <div className="section__i-clean">
                <YSpin progress size="xxs"/>
            </div> :
            <Wrap className={this.state.faded ? 'faded' : ''}>
                <SimpleTable
                    columns={columns}
                    tableHead={this.getTableHead}
                    mapData={this.mapData}
                    sort={this.state.sort}
                    onSortChange={this.handleSortChange}
                    data={data.filter(Boolean)}
                    keyName="title"
                    />
            </Wrap>;
    }
}

SemaphoreView.propTypes = {
    semaphore: propTypes.object,
    tasks: propTypes.oneOfType([
        propTypes.object,
        propTypes.array
    ]),
    fetchTaskList: propTypes.func
};

function mapStateToProps(state, props) {
    const items = get(state, ['tasks', 'items']);
    const tasks = items ?
        ((props.semaphore && props.semaphore.tasks) || []).map(key => get(state, ['tasks', 'items', String(key.task_id)])) :
        { isLoading: true };

    return {
        tasks,
        total: get(state, ['tasks', 'items', 'total'])
    };
}

function mapDispatchToProps(dispatch) {
    return {
        fetchTaskList: props => dispatch(api.fetchTaskList(props))
    };
}

module.exports = connect(mapStateToProps, mapDispatchToProps)(SemaphoreView);
