import React, { Component } from 'react';
import { connect } from 'react-redux';
import PropTypes from 'prop-types';
import styled from 'styled-components';
import { get, find } from 'lodash';
import moment from 'moment';

import { YInput, YButtonGroup, YButton } from 'y-components';

import Modal from '../Modal';
import Owner from '../../common/OwnerSelector';
import { SuggestGroupMulti } from '../../../components/SuggestGroup';

import i18n from '../../i18n';

import API from '../../../api';

const Label = styled.div`
    display: inline-block;
    max-width: 160px;
    width: 219px;
`;

const Row = styled.div`
    display: flex;
    margin-bottom: 16px;
`;

class Edit extends Component {
    constructor(props) {
        super(props);

        this.state = this.defaultState(props);

        this.handlePushSemaphore = this.handlePushSemaphore.bind(this);

        this.handleChange = this.handleChange.bind(this);
        this.handleGroupChange = this.handleGroupChange.bind(this);
        this.handleSaveClick = this.handleSaveClick.bind(this);
        this.handleCancelClick = this.handleCancelClick.bind(this);
        this.handleRadioChange = this.handleRadioChange.bind(this);

        this.getEditContent = this.getEditContent.bind(this);
        this.getCreateContent = this.getCreateContent.bind(this);
    }

    defaultState({ data }) {
        /* eslint complexity: [1, 12] */
        return {
            id: data.id || '',
            name: data.name || '',
            owner: data.owner || '',
            shared: data.shared || [],
            capacity: data.capacity || '',
            auto: data.auto ? ['auto'] : ['persistent'],
            updated: (data.time && data.time.updated) || '',
            created: (data.time && data.time.created) || '',
            event: data.event || ''
        };
    }

    componentWillReceiveProps(newProps) {
        this.setState(
            this.defaultState(newProps),
            this.forceUpdate
        );
    }

    handlePushSemaphore(payload) {
        this.props.onPushSemaphore(payload);
        this.props.pushSemaphore(payload);
    }

    handleGroupChange(groups) {
        this.setState({ shared: groups });
    }

    handleChange(e) {
        if (e.target.value !== this.state[e.target.name]) {
            this.setState({ [e.target.name]: e.target.value });
        }
    }

    handleRadioChange(e, { value, name }) {
        const values = this.state[name];
        const index = values.indexOf(value);

        if (index !== -1) {
            return;
        }

        this.setState({ [name]: [value] });
    }

    handleSaveClick() {
        const { id, shared, capacity, name, owner, auto, event } = this.state;
        const isNew = !id;

        const dataToPush = isNew ?
            { owner, shared, capacity, name } :
            { id, owner, capacity, event, shared };

        if (auto[0]) {
            dataToPush.auto = auto[0] === 'auto';
        }

        this.handlePushSemaphore(dataToPush);
    }

    handleCancelClick() {
        this.setState(this.defaultState(this.props));
    }

    getEditContent() {
        return (<div>
            <Row>
                <Label>Type</Label>
                <YButtonGroup
                    theme="normal"
                    size="s"
                    values={this.state.auto}
                    onChange={this.handleRadioChange}
                    >
                    <YButton name={'auto'} value={'auto'}>Auto</YButton>
                    <YButton name={'auto'} value={'persistent'}>Persistent</YButton>
                </YButtonGroup>
            </Row>
            <Row>
                <Label>Name</Label>
                <div>{this.state.name}</div>
            </Row>
            <Row>
                <Label>Owner</Label>
                <Owner
                    name="owner"
                    onChange={this.handleChange}
                    value={this.state.owner}
                    />
            </Row>
            <Row>
                <Label>Shared with</Label>
                <SuggestGroupMulti
                    value={this.state.shared}
                    name="shared"
                    placeholder={i18n.labels.semaphore.shared}
                    onChange={this.handleGroupChange}
                    />
            </Row>
            <Row>
                <Label>Capacity</Label>
                <YInput
                    name="capacity"
                    size="s"
                    onChange={this.handleChange}
                    value={this.state.capacity}
                    placeholder={i18n.labels.semaphore.capacity}
                    />
            </Row>
            <Row>
                <Label>Description</Label>
                <YInput
                    textarea
                    rows={4}
                    name="event"
                    size="s"
                    onChange={this.handleChange}
                    value={this.state.event}
                    placeholder={i18n.labels.semaphore.event}
                    />
            </Row>
            <Row>
                <Label>Created</Label>
                <div>
                    {moment(this.state.created).format('DD-MM-YYYY HH:MM:SS')}&nbsp;
                    <span className="gray">({moment(this.state.created).fromNow()})</span>
                </div>
            </Row>
            <Row>
                <Label>Updated</Label>
                <div>
                    {moment(this.state.updated).format('DD-MM-YYYY HH:MM:SS')}&nbsp;
                    <span className="gray">({moment(this.state.updated).fromNow()})</span>
                </div>
            </Row>
        </div>);
    }

    getCreateContent() {
        return (<div>
            <Row>
                <Label>Name</Label>
                <YInput
                    name="name"
                    size="s"
                    onChange={this.handleChange}
                    value={this.state.name}
                    placeholder={i18n.labels.semaphore.name}
                    />
            </Row>
            <Row>
                <Label>Owner</Label>
                <Owner
                    name="owner"
                    onChange={this.handleChange}
                    value={this.state.owner}
                    />
            </Row>
            <Row>
                <Label>Shared with</Label>
                <SuggestGroupMulti
                    value={this.state.shared}
                    name="shared"
                    placeholder={i18n.labels.semaphore.shared}
                    onChange={this.handleGroupChange}
                    />
            </Row>
            <Row>
                <Label>Capacity</Label>
                <YInput
                    name="capacity"
                    size="s"
                    onChange={this.handleChange}
                    value={this.state.capacity}
                    placeholder={i18n.labels.semaphore.capacity}
                    />
            </Row>
        </div>);
    }

    render() {
        const isEditMode = Boolean(this.state.id);

        return (
            <Modal
                edit={isEditMode}
                label={this.props.label}
                onSubmit={this.handleSaveClick}
                onCancel={this.handleCancelClick}
                >
                {
                    isEditMode ?
                        this.getEditContent() :
                        this.getCreateContent()
                }
            </Modal>
        );
    }
}

Edit.propTypes = {
    label: PropTypes.object,
    pushSemaphore: PropTypes.func,
    onPushSemaphore: PropTypes.func
};

Edit.defaultProps = {
    label: null,
    onPushSemaphore: () => {}
};

function mapStateToProps(state, ownProps) {
    const data = ownProps.id ? find(
        get(state, ['semaphores', 'items'], []),
        item => {
            return item.id === ownProps.id;
        }
    ) : {};

    return {
        data: Object.assign({}, data)
    };
}

function mapDispatchToProps(dispatch) {
    return {
        pushSemaphore(payload) {
            return dispatch(API.semaphores.pushSemaphore(payload));
        }
    };
}

module.exports = connect(mapStateToProps, mapDispatchToProps)(Edit);
