/* eslint react/jsx-indent-props: 0 */

import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import styled from 'styled-components';
import { get } from 'lodash';

import API from '../../../api';

import { YSpin } from 'y-components';

import Splash from '../../Splash';
import Pagination from '../../Pagination';

import Item from './Item';
import Group from './Group';

const Wrapper = styled.div`
    display: inline-table;
    width: 100%;
    padding: 7px 20px;
    box-shadow: inset 0 1px 0 #EBEBEB;
    background: #eee;
    box-sizing: border-box;
    font-weight: 700;

    > div {
        display: table-cell;
    }
`;

export const Name = styled.div`
    width: 40%;
    word-break: break-all;
    
    > .link {
        display: inline;
    }

    ${props => props.level > 1 && `padding-left: ${(props.level - 1) * 20}px`}
`;

export const Type = styled.div`
    width: 5.5%;
    min-width: 50px;
    vertical-align: middle;
`;

export const Owner = styled.div`
    width: 20%;
    word-break: break-all;
    vertical-align: middle;
`;

export const Shared = styled.div`
    width: 20%;
    word-break: break-all;
    vertical-align: middle;
`;

export const Value = styled.div`
    width: 5%;
    min-width: 60px;
    vertical-align: middle;
`;
export const Capacity = styled.div`
    width: 5%;
    min-width: 60px;
    vertical-align: middle;
`;

export const Edit = styled.div`
    width: 3%;
    vertical-align: middle;
    
    > div {
        min-width: 42px;
    }
`;

const SpinWrap = styled.div`
    .y-spin {
        margin: 10px 20px;
    }
`;

const Relative = styled.div`
    position: relative;
`;

const PageInfo = styled.div`
    display: flex;
    justify-content: space-between;
`;

const Total = styled.div`
    margin: 20px;
    line-height: 28px;
`;

class List extends PureComponent {
    constructor() {
        super(...arguments);

        this.state = {
            page: 1,
            limit: 20
        };

        this.renderGroup = this.renderGroup.bind(this);
        this.handleRangeChange = this.handleRangeChange.bind(this);
    }

    componentDidMount() {
        this.props.fetchSemaphoresList();
    }

    handleRangeChange(e) {
        const isLimitChanged = this.state.limit !== e.limit;
        const state = {
            limit: e.limit,
            page: isLimitChanged ? 1 : e.page
        };

        this.setState(state);

        this.props.onRangeChange(state);
    }

    renderGroup(group, level = 0) {
        const { items: _items, groups: _groups, removeSemaphore } = this.props;

        const { items = [], groups = [] } = group;
        const groupElements = groups.map(groupKey => {
            const group = _groups[groupKey];

            return group ?
                this.renderGroup(group, level + 1) :
                null;
        });

        const itemElements = items.map(itemKey => {
            const item = _items[itemKey];

            return item ?
                <Item
                key={item.id}
                level={level + 1}
                group={group.name || '/'}
                removeSemaphore={removeSemaphore}
                {...item}
                /> :
                null;
        });

        return (
            <Group
                key={group.name}
                isRoot={group.type === 'root'}
                level={level}
                fetchSubtleGroup={this.props.fetchSubtleGroup}
                {...group}
                >
                {groupElements}
                {itemElements}
            </Group>
        );
    }

    render() {
        const { total, isLoading, viewType } = this.props;
        const { page, limit } = this.state;
        const hasNext = total >= limit * page;

        return (
            <Relative>
                {isLoading ? <Splash/> : null}
                <Wrapper>
                    <Name>Name</Name>
                    <Type>Type</Type>
                    <Owner>Owner</Owner>
                    <Shared>Shared</Shared>
                    <Value>Value</Value>
                    <Capacity>Capacity</Capacity>
                    <Edit/>
                </Wrapper>
                <div>
                    {this.props.groups ?
                        this.renderGroup(this.props.groups['/']) :
                        <SpinWrap><YSpin progress size={'xs'}/></SpinWrap>
                    }
                </div>
                <PageInfo>
                    {
                        viewType !== 'groups' && <Pagination
                            limit={this.state.limit}
                            current={this.state.page}
                            hasNext={hasNext}
                            onChange={this.handleRangeChange}
                            size="xs"
                            />
                    }
                    <Total>Total: {total}</Total>
                </PageInfo>
            </Relative>
        );
    }
}

List.propTypes = {
    groups: PropTypes.object,
    items: PropTypes.object,
    total: PropTypes.number,
    isLoading: PropTypes.bool,
    fetchSemaphoresList: PropTypes.func,
    fetchSubtleGroup: PropTypes.func,
    removeSemaphore: PropTypes.func,
    onRangeChange: PropTypes.func,
    viewType: PropTypes.string
};

List.defaultProps = {
    total: 0,
    isLoading: false
};

function mapStateToProps(state) {
    const { groups, items, total, isLoading } = state.semaphores;
    const viewType = get(state, ['user', 'configure', 'semaphores', 'view'], 'groups');

    return {
        groups,
        items,
        total,
        isLoading,
        viewType
    };
}

function mapDispatchToProps(dispatch) {
    return {
        fetchSemaphoresList: group => dispatch(API.semaphores.fetchSemaphoresList(group)),
        fetchSubtleGroup: group => dispatch(API.semaphores.fetchSubtleGroup(group)),
        removeSemaphore: (group, id) => dispatch(API.semaphores.removeSemaphore(group, id))
    };
}

export default connect(mapStateToProps, mapDispatchToProps)(List);
