import React, { Component } from 'react';
import PropTypes from 'prop-types';
import axios, { CancelToken } from 'axios';
import debounce from 'lodash/debounce';

import DataListContainer from '../DataListContainer';

class Suggest extends Component {
    constructor(props) {
        super(props);

        this.state = {
            variants: [],
            value: props.value
        };

        const resources = Array.isArray(props.resources) ?
            props.resources :
            [props.resources];

        this._resources = resources;

        this.handleChange = this.handleChange.bind(this);
        this.fetch = debounce(this.fetch, 50);
    }

    handleChange(e) {
        const { value } = e.target;

        this.setState({
            value
        });

        this.props.onChange(e);
    }

    componentWillReceiveProps(newProps) {
        if (!this._multi && this.props.value !== newProps.value && newProps.value) {
            this.updateData(newProps.value);
        }

        if (this.state.value !== newProps.value && !Array.isArray(newProps.value)) {
            this.setState({
                value: newProps.value
            });
        }
    }

    updateData(value) {
        const result = [];
        const _resources = this._resources;

        if (!value) {
            return this.setState({
                variants: result
            });
        }

        this.props.onLoadingStart();

        _resources.forEach((resource, order) => {
            const { url, name, paramName, title } = resource;

            this.fetch({ url, name, paramName, value }, data => {
                data = data.filter(item => item.tag).map(item => item.tag);

                result.push({
                    order,
                    data,
                    title
                });

                this.setState({
                    variants: result
                });
                this.props.onLoadingEnd();
            });
        });
    }

    fetch(opts, cb) {
        const self = this;
        const url = this.getUrl(opts);
        const headers = this.getHeader();

        if (self.cancel) {
            self.cancel();
        }

        axios
            .get(url, Object.assign({
                cancelToken: new CancelToken(c => {
                    self.cancel = c;
                })
            }, headers || {}))
            .then(result => {
                if (result && result.statusText !== 'OK') {
                    return this.onError(result);
                }

                cb(this.processingResult(result.data, opts));
            })
            .catch(thrown => {
                if (!axios.isCancel(thrown)) {
                    throw new Error(thrown);
                }
            });
    }

    onError(err) {
        this.setState({
            variants: []
        });
        console.warn(err); // eslint-disable-line
    }

    prepare(variants) {
        const result = [];

        variants = variants.sort((a, b) => a.order > b.order);

        variants.forEach(variant => {
            const { data, title } = variant;

            data.forEach(val => {
                result.push({
                    value: val,
                    desc: val,
                    title
                });
            });
        });

        return result;
    }

    processingResult(res, { paramName }) {
        return res.map(i => i[paramName]);
    }

    getUrl({ url, value }) {
        return url.replace('%%value%%', value);
    }

    getHeader() {
        return {};
    }

    render() {
        const { name, placeholder } = this.props;
        const { variants, value } = this.state;

        return (
            <DataListContainer
                name={name}
                mix={this.mix || ''}
                placeholder={placeholder}
                value={value}
                data={this.prepare(variants)}
                onChange={this.handleChange}
                />
        );
    }
}

Suggest.propTypes = {
    onChange: PropTypes.func,
    onLoadingStart: PropTypes.func,
    onLoadingEnd: PropTypes.func,
    value: PropTypes.oneOfType([
        PropTypes.string,
        PropTypes.array
    ]),
    resources: PropTypes.oneOfType([
        PropTypes.array,
        PropTypes.object
    ]),
    name: PropTypes.string,
    placeholder: PropTypes.string
};

Suggest.defaultProps = {
    onLoadingStart: () => {},
    onLoadingEnd: () => {},
    resources: {
        name: 'resources'
    }
};

module.exports = Suggest;
