const sanitize = require('sanitize-html');
// eslint-disable-next-line no-unused-vars
const footerItemTemplate = require('../../views/singleTask/tpl/SingleTaskCustomFooterItemView.hbs');
const footerTableItemView = require('../../views/singleTask/tpl/SingleTaskCustomFooterTableItemView.hbs');
const footerFormattedTable = require('../../views/singleTask/tpl/SingleTaskCustomFooterFormattedTableView.hbs');
const footerPlainTableItemView = require('../../views/singleTask/tpl/SingleTaskCustomFooterPlainTableItemView.hbs');

const TaskFooterBuilder = function (options) {
    this.setRawFooterData(options.data);

    this.processedFooterData = [];
    this.allowedTags = sanitize.defaults.allowedTags.concat(['style', 'input', 'span']);
    this.allowedAttributes = Object.assign(
        {},
        sanitize.defaults.allowedAttributes,
        {
            input: ['name', 'value', 'checked', 'style', 'type'],
            td: ['colspan', 'rowspan'],
            th: ['colspan', 'rowspan'],
            '*': ['style', 'color', 'bgcolor', 'title', 'valign', 'class']
        }
    );
};

_.extend(TaskFooterBuilder.prototype, {/** @lends TaskFooterBuilder.prototype */

    setRawFooterData(data) {
        this.rawFooterData = (data || []);

        if (!(this.rawFooterData instanceof Array)) {
            this.rawFooterData = [this.rawFooterData];
        }
    },

    /**
     * @returns jQuery.Deferred
     */
    buildFooters() {
        const self = this;

        return this.rawFooterData.map(footerData => {
            return footerItemTemplate({
                content: self.getBuildMethod(footerData).call(self, footerData)
            });
        });
    },

    buildPlainTable(data, nested) {
        const headers = data && data.length && Object.keys(data[0]);
        const rows = data.map(function (dataItem) {
            return {
                cells: headers.map(function (key) {
                    return this.buildHtmlSnippet(dataItem[key]);
                }, this)
            };
        }, this);

        return footerPlainTableItemView({
            headKeys: headers,
            rows,
            nested
        });
    },

    buildNestedTable(data, nested) {
        const rows = [];
        const self = this;

        Object.keys(data).forEach(dataItemKey => {
            rows.push({
                key: dataItemKey,
                value: self.getBuildMethod(data[dataItemKey]).call(self, data[dataItemKey], true)
            });
        });

        return footerTableItemView({ rows, nested });
    },

    buildFormattedTable(source, nested) {
        const data = {
            nested,
            header: this.getFormattedTableHeaderKeys(source),
            body: []
        };

        for (let q = 0; q < source.body[source.header[0].key].length; q++) {
            data.body.push(_.map(source.header, header => {
                return _.isNull(source.body[header.key][q]) ?
                    '&nbsp;' :
                    source.body[header.key][q];
            }));
        }

        return footerFormattedTable(data);
    },

    getFormattedTableHeaderKeys(source) {
        let keys = [];

        if (source.header) {
            keys = _.map(source.header, headerObj => {
                return headerObj.title;
            });
        } else {
            keys = Object.keys(source.body);

            this.addHeaderInfoToSource(source);
        }

        return keys;
    },

    buildHtmlSnippet(data) {
        return sanitize(data, {
            allowedTags: this.allowedTags,
            allowedAttributes: this.allowedAttributes
        });
    },

    getBuildMethod(data) {
        let buildMethod = this.buildHtmlSnippet;

        if (data instanceof Array) {
            buildMethod = this.buildPlainTable;
        } else if (data instanceof Object && data !== null) {
            if (this.isFormattedTable(data)) {
                buildMethod = this.buildFormattedTable;
            } else {
                buildMethod = this.buildNestedTable;
            }
        }

        return buildMethod;
    },

    isFormattedTable(source) {
        const fullDescr = _.isArray(source.header) &&
            _.isObject(source.body) &&
            source.header.length === Object.keys(source.body).length;

        return (fullDescr || (_.isObject(source.body) && Object.keys(source.body).length > 0));
    },

    addHeaderInfoToSource(source) {
        source.header = _.map(Object.keys(source.body), headItem => {
            return { key: headItem };
        });
    }
});

module.exports = TaskFooterBuilder;
