/* eslint camelcase: 0 */

const app = require('../app');
// eslint-disable-next-line no-unused-vars
const TaskModel = require('./TaskModel');
const SchedulerTaskModel = require('./SchedulerTaskModel');

/**
 * @class CustomFieldModel
 * @extends Backbone.Model
 */
const CustomFieldModel = Backbone.Model.extend({

    defaults() {
        return CustomFieldModel.getDefaults();
    },

    initialize() {
        this.listenTo(this, 'change:value', this.validateBoth);

        this.dependFrom = null;
        this.dependFromValue = [];

        this.throttleCounter = 0;

        const modifiers = this.get('modifiers');

        if (modifiers && modifiers.format === 'json') {
            this.set('validateHelper', app.getHelperManager().getValidationHelper('json_helper'));
        }
    },

    validateBasedOnProps() {
        const value = this.get('value');
        const result = { success: true };
        const validationProps = this.get('validationProps');

        Object.keys(validationProps).forEach(prop => {
            if (prop === 'required' && validationProps[prop] === true && (value === '' || value === null)) {
                result.success = false;
                result.message = 'That field is required';
            }

            if (prop === 'size' && (String(value).length > validationProps[prop])) {
                result.success = false;
                result.message = ('Should be not longer then ' + validationProps[prop]);
            }
        });

        return result;
    },

    /**
     * @returns {JQueryPromise}
     */
    validateWithHelper() {
        const helper = this.get('validateHelper');
        let result = true;

        if (this.get('type') !== 'block' && typeof helper === 'function') {
            // Helper argument formatted to fit 'unified_helper'
            result = helper({
                parent: this.get('parentModel'),
                fields: [{
                    value: this.get('value'),
                    name: this.get('name')
                }]
            });
        }

        return $.when(result);
    },

    getValidateURL() {
        const parent = this.get('parentModel');

        if (parent instanceof SchedulerTaskModel) {
            return ('/api/v1.0/scheduler/' + parent.getScheduler().get('id') + '/custom/fields');
        }
        return (parent.url() + '/custom/fields');
    },

    /**
     * @description Will validate model 'value' field both with helper (if exist) and validate params
     *              hash. Will resolve on success and reject on validation fail. Also will trigger
     *              'invalid' event on validation fail.
     *
     * @returns {JQueryDeferred}
     */
    validateBoth() {
        /* eslint max-statements: [1, 15] */

        const self = this;
        const parentModel = self.get('parentModel');
        const value = self.get('value');
        const validated = $.Deferred(); // eslint-disable-line
        const staticValidResult = { success: true };

        this.triggerValidationStart(parentModel);

        if (staticValidResult.success === true) {
            this.validateWithHelper(value)
                .done(() => {
                    validated.resolve();

                    self.trigger('valid', value);

                    self.triggerValidationEnd(parentModel);
                })
                .fail(err => {
                    validated.reject(err);

                    self.trigger('invalid', err);

                    self.triggerValidationEnd(parentModel);
                });
        } else {
            validated.reject(staticValidResult);

            self.trigger('invalid', staticValidResult);

            self.triggerValidationEnd(parentModel);
        }

        return validated;
    },

    triggerValidationStart(parentModel) {
        this.throttleCounter++;
        parentModel.trigger('task:validating', 'on');
    },

    triggerValidationEnd(parentModel) {
        this.throttleCounter--;
        if (!this.throttleCounter) {
            parentModel.trigger('task:validating', 'off');
        }
    },

    /**
     * @param {CustomFieldModel} dependFrom
     * @param {String}           dependFromValue
     */
    setDependFrom(dependFrom, dependFromValue) {
        this.dependFrom = dependFrom;
        this.dependFromValue.push(String(dependFromValue));
        this.__available = dependFromValue === dependFrom.get('value');

        this.listenTo(this.dependFrom, 'change:value', this.onDependChange);
    },

    getDependFrom() {
        return this.dependFrom;
    },

    getDependFromValue() {
        return (_.isArray(this.dependFromValue) ? this.dependFromValue : [this.dependFromValue]);
    },

    onDependChange() {
        this.setAvailable();
        this.trigger('toggle', this.__available);
    },

    setAvailable() {
        this.collection.map(item => {
            if (item.dependFrom && item.dependFromValue) {
                item.__available =
                    item.getDependFromValue().includes(String(item.dependFrom.get('value'))) &&
                    item.dependFrom.__available;
            } else {
                item.__available = true;
            }

            return item;
        });
    },

    isAvailable() {
        return !(this.dependFrom && this.dependFromValue) || this.__available;
    }

}, {
    parse(source) {
        const props = CustomFieldModel.getDefaults();

        props.validateHelper = app.getHelperManager().getValidationHelper('unified_helper');

        _.extend(props, source);

        if (typeof source.value !== 'undefined') {
            props.value = source.value;
        }

        if (source.options) {
            props.options = _.extend(props.options, source.options);
        }

        if (source.sub_fields) {
            const subs = source.sub_fields;

            props.subFields = subs;
        }

        return props;
    },

    getDefaults() {
        return {
            name: '',
            value: null,
            title: '',
            values: [], // For selects
            task_id: 0,
            options: {},
            template: '',
            subFields: {}, /** {String, String[]} */ // for boolean_sub and select_sub?
            // subActiveVal: true,
            buildHelper: null,
            validateHelper: null,
            validationProps: {
                required: false
            }
        };
    },

    EDIT_HELPER_NAME: 'buildHelper'
});

module.exports = CustomFieldModel;
