/**
 * @class DateTimeIntervalModel
 * @extends Backbone.Model
 */
const DateTimeIntervalModel = Backbone.Model.extend({

    defaults: {
        from: null,
        till: null,

        // Offset will be set relatively to .till
        offset: '' // String %value%_%measure%
    },

    options: {
        dateOnly: false
    },

    initialize(props, options) {
        this.options.dateOnly = ((options && options.dateOnly) ? options.dateOnly : this.options.dateOnly);
    },

    setFrom(from) {
        if (from === null || (from && moment.isMoment(from))) {
            this.set('from', from);
        }
    },

    dropTime(date) {
        if (date !== null && this.options.dateOnly) {
            date.hours(0);
            date.minutes(0);
            date.seconds(0);
        }

        return date;
    },

    setTill(till) {
        if (till === null || (till && moment.isMoment(till))) {
            this.set('till', till);
        }
    },

    setOffset(offset) {
        this.set('offset', offset);
    },

    setDTInterval(from, till, offset) {
        this.setFrom(from);
        this.setTill(till);
        this.setOffset(offset);
    },

    getTill() {
        let till = this.get('till');

        if (till === DateTimeIntervalModel.NOW) {
            till = moment();
        }

        return till;
    },

    getFrom() {
        let from = this.get('from');
        const till = this.getTill();
        const offset = this.get('offset');

        if (!from) {
            if (till) {
                from = till.clone().subtract(offset.size, offset.measure);
            }
        }

        if (this.options.dateOnly) {
            from = this.dropTime(from);
        }

        return from;
    },

    stringify() {
        return DateTimeIntervalModel.stringify(this.toJSON());
    },

    serialize() {
        const serialized = {};
        const interval = this.toJSON();

        serialized.from = this.serializePart(interval.from);
        serialized.till = this.serializePart(interval.till);

        return serialized;
    },

    serializePart(date) {
        return DateTimeIntervalModel.serializePart(date);
    },

    parse(data) {
        let parsed = {
            from: null,
            till: null,
            offset: ''
        };

        if (typeof data === 'string') {
            parsed = DateTimeIntervalModel.parseStringifiedData(data).attributes;
        } else if (typeof data === 'object') {
            parsed = DateTimeIntervalModel.parseCustomInterval(data);
        }

        return parsed;
    }
}, {

    /**
     * @param {object} interval
     * @returns {string}
     */
    stringify(interval) {
        let result = '';

        if (interval.from && interval.till) {
            result += DateTimeIntervalModel.serializePart(interval.from);
            result += '..';
            result += DateTimeIntervalModel.serializePart(interval.till);
        } else if (interval.offset && interval.offset !== DateTimeIntervalModel.CSTM) {
            const offsetParams = DateTimeIntervalModel.parseIntervalDescription(interval.offset);

            result += DateTimeIntervalModel.serializePart(moment().subtract(offsetParams.size, offsetParams.measure));
            result += '..';
            result += DateTimeIntervalModel.serializePart(moment());
        }

        return (result === '..' ? '' : result);
    },

    humanify(interval) {
        if (interval.offset === DateTimeIntervalModel.CSTM) {
            return DateTimeIntervalModel.stringify(interval);
        }
        return interval.offset;
    },

    serializePart(date) {
        if (date === null) {
            return '';
        }
        return date.toISOString();
    },

    parseIntervalDescription(intervalDescription) {
        const parsed = typeof intervalDescription === 'string' ?
            intervalDescription.split('_') :
            [DateTimeIntervalModel.CSTM];
        let description = {
            size: 0,
            measure: null
        };

        if (parsed[0] && !parsed[1] && parsed[0] !== DateTimeIntervalModel.CSTM) {
            // Default measure
            description.size = parseInt(parsed[0], 10);
            description.measure = DateTimeIntervalModel.DEFAULT_MEASURE;
        } else if (parsed[0] && parsed[1]) {
            // Measure specified
            description.size = parseInt(parsed[0], 10);
            description.measure = parsed[1];
        } else if (parsed[0] === DateTimeIntervalModel.CSTM) {
            // Custom interval
            description = null;
        }

        return description;
    },

    parseStringifiedData(data) {
        let parsed = null;

        if (data.includes('..')) {
            // Hardcoded values in ISO 8601 format

            parsed = data.split('..');

            parsed = new DateTimeIntervalModel({
                from: moment(parsed[0]),
                till: moment(parsed[1]),
                offset: DateTimeIntervalModel.CSTM
            });
        } else {
            // Relative to NOW values

            parsed = new DateTimeIntervalModel({
                from: null,
                till: null,
                offset: data
            });
        }

        return parsed;
    },

    parseCustomInterval(data) {
        const parsed = data;

        if (data.offset === DateTimeIntervalModel.CSTM) {
            parsed.offset = DateTimeIntervalModel.CSTM;

            if (_.isString(data.from)) {
                parsed.from = moment(data.from);
            }

            if (_.isString(data.till)) {
                parsed.till = moment(data.till);
            }
        }

        return parsed;
    }
});

DateTimeIntervalModel.DEFAULT_MEASURE = 'days';
DateTimeIntervalModel.MEASURES = ['days', 'months', 'minutes', 'hours'];
DateTimeIntervalModel.CSTM = 'CSTM';
DateTimeIntervalModel.NOW = 'NOW';

module.exports = DateTimeIntervalModel;
