/* eslint camelcase: 0 */
/* eslint max-statements: [1, 15] */
/* eslint complexity: [1, 15] */

const app = require('../app');
const Timer = require('../helpers/utils/Timer');
const router = require('../router');
const error = require('../error');
const TaskModel = require('./TaskModel');
const TasksCollection = require('../collection/TasksCollection');
// eslint-disable-next-line no-unused-vars
const BasicTaskLayout = require('../views/tasks/tasksGrid/row/BasicTaskLayout');
const SchedulerTaskModel = require('./SchedulerTaskModel');

/**
 * @class SchedulerModel
 * @extends TaskModel
 *
 * @property {TaskModel} task
 */
const SchedulerModel = TaskModel.extend({

    defaults() {
        return _.clone(SchedulerModel.PROP_SET);
    }, /** All default props will be set in {@link SchedulerModel#parse} method */

    url() {
        const badeURL = '/api/v1.0/scheduler';

        return (this.get('id') ? (badeURL + '/' + this.get('id')) : badeURL);
    },

    initialize() {
        this.listenTo(this, 'actionSuccess', this.fetch);

        this.fetchChildTasks = _.memoize(this.fetchChildTasks);
        this.fetchChildOnceTasks = _.memoize(this.fetchChildOnceTasks);
        this.fetchTaskContext = _.memoize(this.fetchTaskContext);
    },

    getType() {
        return this.task.get('type');
    },

    toJSON() {
        const task = this.getTask();
        const json = _.pick(this.attributes, [
            'id',
            'time',
            'owner',
            'author',
            'status',
            'rights',
            'schedule',
            'description'
        ]);

        json.task = (task ? task.toJSON() : {});

        return json;
    },

    /**
     * @param {Object} response
     * @param {Object} options
     *
     * @returns {Object|Null} Will return parsed object in all cases except when parsing is off
     *                        by setting options.parse to FALSE.
     */
    parse(response, options) {
        if (options && options.parse === false) {
            return null;
        }

        response = (typeof response === 'undefined' ? { task: {}} : response);

        const parsed = _.extend({}, SchedulerModel.PROP_SET, response);

        if (parsed.task && parsed.task.last) {
            this.last = parsed.task.last;
        }

        parsed.schedule = this.parseSchedule(parsed.schedule);
        parsed.time = this.parseTime(parsed.time);

        this.task = new SchedulerTaskModel(
            Object.assign({}, { last: this.last }, parsed.task),
            { parse: true }
        );
        this.task.setScheduler(this);

        delete parsed.task;

        return parsed;
    },

    parseTime(toParse) {
        return ['next', 'last', 'created', 'updated'].reduce((result, item) => {
            result[item] = Timer.toLocal(toParse[item]).toISOString();

            return result;
        }, {});
    },

    parseSchedule(toParse) {
        const schedule = {
            sequential_run: (toParse && _.isBoolean(toParse.sequential_run) ? toParse.sequential_run : false),
            fail_on_error: (toParse && _.isBoolean(toParse.fail_on_error) ? toParse.fail_on_error : false),
            repetition: null,
            start_time: ((toParse && toParse.start_time) ? Timer.toLocal(toParse.start_time).toISOString() : null),
            retry: {}
        };

        if (!toParse) {
            return schedule;
        }

        if (toParse.repetition && _.isNumber(toParse.repetition.interval)) {
            schedule.repetition = { interval: toParse.repetition.interval };
        }

        if (toParse.repetition && _.isArray(toParse.repetition.weekly)) {
            schedule.repetition = { weekly: toParse.repetition.weekly };
        }

        if (toParse.retry) {
            if (_.isNumber(toParse.retry.interval)) {
                schedule.retry.interval = toParse.retry.interval;
            }

            if (_.isBoolean(toParse.retry.ignore)) {
                schedule.retry.ignore = toParse.retry.ignore;
            }
        } else {
            schedule.retry.ignore = true;
        }

        return schedule;
    },

    duplicate() {
        const self = this;

        this.set('_duplicating', true);

        return $.ajax({
            url: '/api/v1.0/scheduler',
            type: 'POST',
            data: JSON.stringify({
                source: this.get('id')
            }),
            dataType: 'json',
            contentType: 'application/json'
        }).done(response => {
            self.set('_duplicating', false);

            const newScheduler = new SchedulerModel();

            newScheduler.set(SchedulerModel.prototype.parse.call(newScheduler, response));
            app.setDraftScheduler(newScheduler);
        }).fail(err => {
            self.set('_duplicating', false);

            error.fromXHR(err);
        });
    },

    validateGeneralFields() {
        const errors = TaskModel.prototype.validateGeneralFields.call(this.task);
        const schedule = this.get('schedule');

        if (schedule && schedule.repetition && schedule.repetition.weekly && !schedule.repetition.weekly.length) {
            const err = {
                field: 'retry',
                message: 'One or more fields should be checked'
            };

            errors.push(err);
            this.trigger('invalid:retry', err);
        }

        return errors;
    },

    setAsDraft() {
        app.setDraftScheduler(this);
    },

    getTask() {
        return this.task;
    },

    fetchCustomParameters() {
        return this.task.fetchCustomParameters();
    },

    fetchTaskContext() {
        return this.task.fetchContext();
    },

    /**
     * @returns {JQueryDeferred}
     */
    fetchChildTasks() {
        this.childTasks = new TasksCollection([], { scheduler: this.get('id'), owner: '', hidden: true });

        return this.childTasks.update();
    },

    fetchChildOnceTasks() {
        this.childOnceTasks = new TasksCollection([], { scheduler: '-' + this.get('id'), owner: '', hidden: true });

        return this.childOnceTasks.update();
    },

    serialize() {
        const serialized = this.toJSON();
        const time = this.get('time');
        const schedule = this.get('schedule');

        if (this.task) {
            serialized.task = _.clone(this.task.attributes);
        }

        serialized.start_time = (schedule.start_time ? Timer.serializeTimeMark(schedule.start_time) : null);

        ['next', 'last', 'created', 'updated'].forEach(item => {
            serialized[item] = Timer.serializeTimeMark(time[item] ? time[item] : '');
        });

        return serialized;
    },

    setWeeklyRepetition(days) {
        const schedule = this.get('schedule');

        schedule.repetition = {
            weekly: days
        };

        this.setSchedule(schedule);
    },

    setRetryInterval(interval) {
        const schedule = this.get('schedule');

        schedule.retry = {
            interval,
            ignore: false
        };

        this.setSchedule(schedule);
    },

    setIntervalRepetition(interval) {
        const schedule = this.get('schedule');

        schedule.repetition = {
            interval
        };

        this.setSchedule(schedule);
    },

    setNoneRepetition() {
        const schedule = this.get('schedule');

        schedule.repetition = null;

        this.setSchedule(schedule);
    },

    setSchedule(schedule) {
        this.set('schedule', schedule);
        this.trigger('change:schedule');
    },

    setNextRun(nextRun) {
        nextRun = (nextRun ? moment(nextRun) : null);

        const schedule = this.get('schedule');

        schedule.start_time = (nextRun ? nextRun.utc().toISOString() : null);

        this.set('schedule', schedule);
        this.trigger('change:schedule');
    },

    /**
     * @param {string} rawTime
     */
    setNextRunTime(rawTime) {
        /* eslint camelcase: 0 */

        if (rawTime) {
            const matched = rawTime.match(/^([0-9]{2}):([0-9]{2})$/);
            const schedule = this.get('schedule');

            if (!matched || parseInt((matched[1]) > 23 || parseInt(matched[2], 10) > 59, 10)) {
                this.trigger('invalid:startDateTime', {
                    field: 'startDateTime',
                    message: 'Next run time is not valid'
                });

                return false;
            }

            const st = moment(schedule.start_time);

            st.hours(matched[1]);
            st.minutes(matched[2]);

            this.setNextRun(st);
        }
    },

    /**
     * @param {string} rawDate
     * @param {boolean} zeroBased
     */
    setNextRunDate(rawDate, zeroBased) {
        zeroBased = (_.isBoolean(zeroBased) ? zeroBased : true);

        if (rawDate) {
            const matched = rawDate.match(/^([0-9]{2})\.([0-9]{2})\.([0-9]{4})$/);
            const schedule = this.get('schedule');
            let st = null;

            if (!matched ||
                parseInt(matched[1], 10) > 31 ||
                parseInt(matched[2], 10) > 12 ||
                parseInt(matched[3], 10) < moment().year()) {
                this.trigger('invalid:startDateTime', {
                    field: 'startDateTime',
                    message: 'Next run date is not valid'
                });

                return false;
            }

            st = moment(schedule.start_time);
            st.date(parseInt(matched[1], 10));
            st.month(parseInt(matched[2], 10) - (zeroBased ? 0 : 1));
            st.year(parseInt(matched[3], 10));

            this.setNextRun(st);
        }
    },

    setNonZeroBasedNextRunDate(rawDate) {
        this.setNextRun(rawDate, false);
    },

    getAction() {
        const status = this.get('status');
        const action = status === SchedulerModel.STATUS.WAITING || status === SchedulerModel.STATUS.WATCHING ?
            'stop' :
            'start';

        return action;
    },

    startAction() {
        const action = this.getAction();
        const SchedulersCollection = require('../collection/SchedulersCollection');

        return SchedulersCollection.batchAction(action, [this]);
    },

    startActionOnce() {
        if (this.editMode) {
            /* Триггерим событие для валидации и последующей отбработки действия про создание задачи */
            this.trigger('scheduler:runOnce');
            Backbone.trigger('scheduler:runOnce:finished');
        } else {
            /* Нет формы редактирвания, данные валидны. Просто сохряняем таск */
            const self = this;
            const task = this.task;

            Object.keys(TaskModel.PROP_SET).forEach(item => {
                task.unset(item, { silent: true });
            });

            return task.save({
                scheduler_id: self.id,
                type: null
            })
                .then(() => {
                    const status = task.get('status');
                    const action = status === TaskModel.STATUS.WAITING || status === TaskModel.STATUS.WATCHING ?
                        'stop' :
                        'start';
                    const TasksCollection = require('../collection/TasksCollection');

                    Backbone.trigger('scheduler:runOnce:finished');

                    TasksCollection.batchAction(action, [task]).done(() => {
                        router.navigate('/task/' + task.get('id') + '/view', { trigger: true });
                    });
                })
                .fail(e => {
                    Backbone.trigger('scheduler:runOnce:finished');

                    if (!(app.error.currentView && app.error.currentView.$el)) {
                        const reason = (e && (e.reason || (e.responseJSON && e.responseJSON.reason))) ||
                        'Some fields are invalid';

                        error.message(reason, {
                            reload: false
                        });
                    }
                });
        }
    },

    canBeStopped() {
        const status = this.get('status');

        return this.canBeEdited() &&
            (status === SchedulerModel.STATUS.WAITING || status === SchedulerModel.STATUS.WATCHING);
    },

    canBeStarted() {
        return (!this.canBeStopped() && this.canBeStartedOnce() && this.canBeEdited());
    },

    canBeStartedOnce() {
        return this.get('status') !== SchedulerModel.STATUS.DELETED;
    },

    canBeEdited() {
        return this.get('rights') === 'write';
    },

    canBeClonned() {
        return !this.get('_duplicating');
    }
});

SchedulerModel.PROP_SET = {

    // Same as task
    time: {},
    owner: '',
    author: '',
    status: '',
    rights: [], // WARN: String ?
    description: '',

    // Scheduler specific
    // task: null,
    schedule: null,

    // Utility
    selected: false,

    task: TaskModel.PROP_SET
};

SchedulerModel.STATUS = {
    WAITING: 'WAITING',
    WATCHING: 'WATCHING',
    DELETED: 'DELETED',
    UNKNOWN: 'UNKNOWN',
    FAILURE: 'FAILURE',
    STOPPED: 'STOPPED'
};

module.exports = SchedulerModel;
