/* eslint camelcase: 0 */
/* eslint max-statements: [1, 21] */
/* eslint no-useless-escape: 0 */

const app = require('../app');
const error = require('../error');
// Const UserModel = require('./user'); // @TODO remove fast fix for USER:HIGH priorities
const ClientModel = require('./clients/ClientModel');
const notification = require('../notification');
const ChildTaskModel = require('./ChildTaskModel');
const ChildTasksCollection = require('../collection/ChildTasksCollection');
const NotificationModel = require('./NotificationModel');
const NotificationsCollection = require('../collection/NotificationsCollection');
const SemaphoresModel = require('./TaskRequirementsSemaphoresModel');
const SemaphoresCollection = require('../collection/TaskRequirementsSemaphoresCollection');
const TaskHistoryItemsCollection = require('../collection/TaskHistoryItemsCollection');

/**
 * @class TaskModel
 * @extends Backbone.Model
 *
 * @requires moment.js
 */
const TaskModel = Backbone.Model.extend({

    defaults() {
        return _.clone(TaskModel.PROP_SET);
    }, /** All default props will be set in {@link TaskModel#parse} method */

    initialize() {
        this.listenTo(this, 'actionSuccess', this.fetch);

        this.editMode = false;
        this.fetchCustomParameters = _.memoize(this.fetchCustomParameters);
    },

    set(attrs, options) {
        if (attrs.selected === null) {
            if (this.attributes.selected !== null) {
                attrs.selected = this.attributes.selected;
            }
        }

        return Backbone.Model.prototype.set.call(this, attrs, options);
    },

    url() {
        const badeURL = '/api/v1.0/task';

        return (this.get('id') ? (badeURL + '/' + this.get('id')) : badeURL);
    },

    getValidateURL() {
        return (this.url() + '/custom/fields');
    },

    validateCustomFields() {
        const validated = [];
        const customFields = this.model.get('customFields');

        customFields.forEach(function (customFieldName) {
            validated.push(this.validateCustomField(customFieldName));
        });

        return jQuery.when.apply(null, validated);
    },

    validateCustomField(fieldName) {
        const fieldData = this.getCustomFieldData(fieldName);
        const value = this.get(fieldName);
        const validPromise = $.Deferred(); // eslint-disable-line

        if (fieldData === null) {
            validPromise.resolve(true);
        } else {
            if (fieldData.context.checker) { // eslint-disable-line
                const helper = app.getHelperManager().getHelper('falsyHelper'/* fieldData.context.checker */);

                if (helper && typeof helper === 'function') {
                    helper(value)
                        .success(function () {
                            validPromise.resolve(arguments);
                        })
                        .fail(function () {
                            validPromise.reject(arguments);
                        });
                } else {
                    validPromise.reject({
                        message: 'Necessary validate helper is not provided'
                    });
                }
            } else {
                validPromise.resolve(true);
            }
        }

        return validPromise;
    },

    validateGeneralFields() {
        /* eslint complexity: [1, 6] */

        const errors = [];
        const owner = this.validateOwner();
        const description = this.validateDescription();
        const killTimeout = this.validateKillTimeout();
        const diskSpace = this.validateDiskSpace();

        if (owner.message) {
            errors.push(owner);
        }

        if (description.message) {
            errors.push(description);
        }

        if (killTimeout.message) {
            errors.push(killTimeout);
        }

        if (diskSpace.message) {
            errors.push(diskSpace);
        }

        if (errors.length > 0) {
            this.trigger('invalid', errors);
        }

        return errors;
    },

    validateKillTimeout() {
        let error = {};
        const timeout = this.get('kill_timeout');

        if (_.isUndefined(timeout)) {
            error = {
                field: 'kill_timeout',
                message: 'That field is required'
            };
        } else if (!(new RegExp('^[0-9]+ ?[hms]?$', 'i')).test(String(timeout))) {
            error = {
                field: 'kill_timeout',
                message: 'Not valid'
            };
        }

        if (error.message) {
            this.trigger('invalid:kill_timeout', error);
        }

        return error;
    },

    validateDescription() {
        let error = {};

        if (!this.get('description')) {
            error = {
                field: 'description',
                message: 'That field is required'
            };

            this.trigger('invalid:description', error);
        }

        return error;
    },

    validateOwner() {
        let error = {};

        if (!this.get('owner')) {
            error = {
                field: 'owner',
                message: 'That field is required'
            };

            this.trigger('invalid:owner', error);
        }

        return error;
    },

    /**
     * @description Will validate disk space. Also can be used to validate disk space without setting it as a model
     *              property. In that case need value need to be passed as argument.
     *
     * @param {Object} requirements
     * @param {String} requirements.disk_space
     *
     * @returns {{}}
     */
    validateDiskSpace(requirements) {
        let error = {};
        const req = requirements || this.get('requirements');

        if (!req.disk_space) {
            error = {
                field: 'disk_space',
                message: 'That field is required'
            };
        } else if (!(new RegExp('^[0-9]+(\.[0-9]{1,2})? ?((GB)|(KB)|(MB)|(TB))?$', 'i')).test(req.disk_space)) {
            error = {
                field: 'disk_space',
                message: 'Not valid'
            };
        }

        if (error.message) {
            this.trigger('invalid:disk_space', error);
        }

        return error;
    },

    /**
     * @description Will validate RAM space. Also can be used to validate disk space without setting it as a model
     *              property. In that case need value need to be passed as argument.
     *
     * @param {Object} requirements
     * @param {String} requirements.ram
     *
     * @returns {{}}
     */
    validateRamSpace(requirements) {
        let error = {};
        const req = requirements || this.get('requirements');

        if (!req.ram) {
            error = {
                field: 'ram',
                message: 'That field is required'
            };
        } else if (!(new RegExp('^[0-9]+(\.[0-9]{1,2})? ?((GB)|(KB)|(MB)|(TB))?$', 'i')).test(req.ram)) {
            error = {
                field: 'ram',
                message: 'Not valid'
            };
        }

        if (error.message) {
            this.trigger('invalid:ram', error);
        }

        return error;
    },

    validateRamDriveSpace(requirements) {
        let error = {};
        const req = requirements || this.get('requirements');

        if (!(new RegExp('^[0-9]+(\.[0-9]{1,2})? ?((GB)|(KB)|(MB)|(TB))?$', 'i')).test(req.ramdrive)) {
            error = {
                field: 'ramdrive',
                message: 'Not valid'
            };
        }

        if (error.message) {
            this.trigger('invalid:ramdrive', error);
        }

        return error;
    },

    validateCores(requirements) {
        let error = {};
        const req = requirements || this.get('requirements');
        const cores = Number(req.cores);

        if (isNaN(cores) || cores <= 0 || cores !== Math.round(cores)) {
            error = {
                field: 'cores',
                message: 'Not valid'
            };
        }

        if (error.message) {
            this.trigger('invalid:cores', error);
        }

        return error;
    },

    validateClientTags(requirements) {
        let error = {};
        const req = requirements || this.get('requirements');

        if (!req.client_tags || !req.client_tags.length) {
            error = {
                field: 'clientTags',
                message: 'Not valid'
            };
        }

        if (error.message) {
            this.trigger('invalid:clientTags', error);
        }

        return error;
    },

    /**
     * @param fieldName
     * @returns {Object|Null}
     */
    getCustomFieldData(fieldName) {
        let data;
        const allData = this.get('customFields');

        if (allData) {
            data = _.find(allData, item => {
                return item.context.name === fieldName;
            });
        }

        return data ? data : null;
    },

    /**
     * @returns {$.Deferred}
     */
    toggleImportant() {
        return this.save({
            important: !this.get('important')
        }).fail(jqXhr => {
            error.fromXHR(jqXhr);
        });
    },

    revertToValidState() {
        const attrs = this.prevAttributes();

        this.set(attrs, { silent: true });
    },

    setSelected(isSelected, options) {
        options = _.extend({ silent: false, isBatch: false }, options);

        this.set({
            selected: (typeof isSelected === 'undefined' ? !this.get('selected') : Boolean(isSelected))
        }, {
                silent: Boolean(options.silent),
                isBatch: Boolean(options.isBatch)
            });
    },

    toggleSelected(options) {
        this.setSelected(!this.get('selected'), options);
    },

    setSelectedSilent(isSelected) {
        this.setSelected(isSelected, {
            silent: true
        });
    },

    setPriorityClass(valueToSet) {
        const priority = this.get('priority');

        priority.class = valueToSet;

        this.set('priority', priority);
    },

    setPrioritySubClass(valueToSet) {
        const priority = this.get('priority');

        priority.subclass = valueToSet;

        this.set('priority', priority);
    },

    setAsDraft() {
        app.setDraftTask(this);
    },

    /**
     * @description Will start/stop task depends on it's current state
     * @returns {*|JQueryDeferred}
     */
    taskAction() {
        const TasksCollection = require('../collection/TasksCollection');
        let action = this.canBeExecuted() && TasksCollection.BATCH_OPERATIONS.START;

        if (this.canBeResumed()) {
            action = TasksCollection.BATCH_OPERATIONS.RESUME;
        }

        return TasksCollection.batchAction(action, [this]);
    },

    /**
     * @description Will start/stop task depends on it's current state
     * @returns {*|JQueryDeferred}
     */
    taskStopAction() {
        const TasksCollection = require('../collection/TasksCollection');
        const action = this.canBeStopped() && TasksCollection.BATCH_OPERATIONS.STOP;

        return TasksCollection.batchAction(action, [this]);
    },

    /**
     * @description Will start/suspend task depends on it's current state
     * @returns {*|JQueryDeferred}
     */
    taskSuspend() {
        const TasksCollection = require('../collection/TasksCollection');

        return TasksCollection.batchAction(TasksCollection.BATCH_OPERATIONS.SUSPEND, [this]);
    },

    toJSON() {
        /* eslint complexity: [1, 12] */

        const json = _.clone(this.attributes);

        if (_.isArray(json.customFields)) {
            json.custom_fields = json.customFields
                .filter(field => {
                    return !field.output;
                }).map(field => {
                    return { name: field.name, value: field.value };
                });
        }

        if (!json.children || json.children.count) {
            json.children = {};
        } else {
            const children = this.get('children');

            if (children && children.toJSON) {
                json.children = children.toJSON();
            } else {
                json.children = {};
            }
        }

        if (json.notifications instanceof NotificationsCollection) {
            json.notifications = json.notifications.toJSON();
        }

        if (json.semaphores instanceof SemaphoresCollection) {
            json.requirements = json.requirements || {};
            json.requirements.semaphores = json.semaphores.toJSON();
        }

        if (json.requirements && json.requirements.ramdrive && !json.requirements.ramdrive.type) {
            json.requirements.ramdrive = {
                type: 'tmpfs',
                size: json.requirements.ramdrive
            };
        }

        delete json.semaphores;
        delete json.selected;
        delete json.customFields;
        delete json.customFieldsMode;

        return json;
    },

    /**
     * @param {Object} response
     * @param {Object} options
     *
     * @returns {Object|Null} Will return parsed object in all cases except when parsing is off
     *                        by setting options.parse to FALSE.
     */
    parse(response, options) {
        if (options && options.parse === false) {
            return null;
        }

        response = (typeof response === 'undefined' ? {} : response);

        const parsed = _.extend({}, TaskModel.PROP_SET, response);

        parsed.audit = this.parseAudit(response.audit);
        parsed.release = this.parseReleaseInfo(response.release);
        parsed.children = this.parseChildren(response.children);
        parsed.execution = this.parseExec(response.execution);
        parsed.notifications = this.parseNotifications(response.notifications);
        parsed.semaphores = this.parseSemaphores((response.requirements || {}).semaphores);

        if (parsed.requirements.ramdrive && parsed.requirements.ramdrive.type) {
            parsed.requirements.ramdrive = parsed.requirements.ramdrive.size;
        }

        if (this.get('audit')) {
            delete parsed.audit;
        }

        return parsed;
    },

    parseReleaseInfo(releaseInfo) {
        if (releaseInfo && releaseInfo.last_release && releaseInfo.last_release.message) {
            const removeCaretReg = /(?:\r\n|\r|\n)/g;

            releaseInfo.last_release.message = releaseInfo.last_release.message.replace(removeCaretReg, '<br />');
        }

        return releaseInfo;
    },

    parseExec(execution) {
        return _.extend({
            action: '',
            current: 0,
            estimated: 0,
            client: {
                url: '',
                id: ''
            }
        }, execution);
    },

    parseChildren(rawChildren) {
        let parsed = null;

        rawChildren = rawChildren || [];

        if (rawChildren) {
            if (rawChildren instanceof Array) {
                parsed = new ChildTasksCollection();

                rawChildren.forEach(child => {
                    Object.keys(child).forEach(childId => {
                        parsed.add(new ChildTaskModel({
                            id: childId,
                            status: child[childId]
                        }));
                    });
                });
            } else if (rawChildren.count) {
                parsed = rawChildren;
            }
        }

        return parsed;
    },

    parseNotifications(data) {
        const notifications = new NotificationsCollection();

        if (!(data instanceof Array)) {
            data = NotificationsCollection.prototype.parse.call(null, data);
        }

        data.forEach(notification => {
            notifications.add(new NotificationModel(notification));
        });

        return notifications;
    },

    parseSemaphores(data) {
        const semaphores = new SemaphoresCollection();

        if (data && data.acquires && data.acquires.length) {
            data.acquires.forEach(semaphore => {
                semaphores.add(new SemaphoresModel({ semaphore, release: data.release }));
            });
        } else {
            semaphores.add(new SemaphoresModel({}));
        }

        return semaphores;
    },

    parseAudit(auditData) {
        const TaskHistoryItemsCollection = require('../collection/TaskHistoryItemsCollection');
        const audit = this.get('audit');

        if (audit instanceof TaskHistoryItemsCollection) {
            return audit;
        }
        return new TaskHistoryItemsCollection([], {
            task: this,
            unfetchedLength: (auditData ? auditData.count : 0)
        });
    },

    increasePriority() {
        const TasksCollection = require('../collection/TasksCollection');

        const current = this.get('priority');
        const updated = {};
        const subclass = this.getIncreasedPriorityValue('SUBCLASS', current.subclass);

        if (subclass) {
            updated.subclass = subclass;
            updated.class = current.class;
        } else {
            updated.subclass = TaskModel.PRIORITY.SUBCLASS[0];
            updated.class = this.getIncreasedPriorityValue('CLASS', current.class);
        }

        if (updated.class && updated.subclass) {
            const result = TasksCollection.batchAction(TasksCollection.BATCH_OPERATIONS.INC_PRIOR, [this]);

            result.then(() => {
                const message = ('Task #' + this.get('id') +
                    ' priority has been increased to ' +
                    updated.class.toUpperCase() + ':' +
                    updated.subclass.toUpperCase());

                this.set('priority', updated);

                notification.message(message, {
                    reload: false,
                    autoClose: true
                });

                this.fetch();
            });

            return result;
        }
    },

    getIncreasedPriorityValue(type, current) {
        return TaskModel.PRIORITY[type][TaskModel.PRIORITY[type].indexOf(current) + 1];
    },

    getResourcesQnt() {
        return this.getCountableAttrQnt(this.get('resources'));
    },

    getContextQnt() {
        const contextCount_sdk1 = this.getCountableAttrQnt(this.get('context'));
        const contextCount_sdk2 = this.getCountableAttrQnt(this.get('input_parameters')) + contextCount_sdk1;

        return Number(this.get('sdk_version')) === 1 ?
            contextCount_sdk1 :
            contextCount_sdk2;
    },

    getHistoryItemsQnt() {
        return this.get('audit').getLength();
    },

    getLogsQnt() {
        return TaskModel.STATUSES.EXECUTE.indexOf(this.get('status')) !== -1;
    },

    getHostsQnt() {
        return this.getCountableAttrQnt(this.get('hosts'));
    },

    getDependantQnt() {
        return this.getCountableAttrQnt(this.get('dependant'));
    },

    getDependOnQnt() {
        return this.getCountableAttrQnt(this.get('requirements').resources);
    },

    getChildrenQnt() {
        return this.getCountableAttrQnt(this.get('children'));
    },

    getType() {
        return this.get('type');
    },

    getCountableAttrQnt(attrValue) {
        let qnt = 0;

        if (attrValue) {
            if (attrValue instanceof Backbone.Collection) {
                qnt = attrValue.length;
            } else if (attrValue.count) {
                qnt = attrValue.count;
            } else {
                qnt = (_.difference(Object.keys(attrValue), ['url', 'count']) || []).length;
            }
        }

        return qnt;
    },

    mapToCustomFields(customFields) {
        customFields = customFields || [];

        return _.map(customFields, function (field) {
            const currVal = this.getCustomFieldValue(field.name);

            if (typeof currVal !== 'undefined') {
                field.value = currVal;
            }

            return field;
        }, this);
    },

    fetchAudit() {
        const self = this;

        return this.fetchAdditionalData('audit').done(audit => {
            self.set('audit', new TaskHistoryItemsCollection(audit));
        });
    },

    fetchRelatedHosts() {
        const self = this;

        return this.fetchAdditionalData('audit/hosts').done(relatedHosts => {
            if (_.isArray(relatedHosts)) {
                self.set('relatedHosts', relatedHosts);
            }
        });
    },

    fetchContext() {
        const self = this;
        const fetched = $.Deferred(); // eslint-disable-line
        const customFields = this.fetchCustomParameters();

        this.fetchAdditionalData('context')
            .done((response, state, xhr) => {
                if (xhr.readyState === 4 && xhr.status >= 200 && xhr.status < 300) {
                    self.set('context', response);
                    customFields.always(() => {
                        fetched.resolve(response, state, xhr);
                    });
                }
            })
            .fail(function () {
                fetched.reject(arguments[0], arguments[1], arguments[2]);
            });

        return fetched;
    },

    fetchDependant() {
        const TasksCollection = require('../collection/TasksCollection');
        const self = this;
        const dependant = new TasksCollection();
        const prevAttributes = self._previousAttributes;

        return this.fetchAdditionalData('dependant', { children: true, hidden: true }).done(data => {
            data.forEach(childTaskData => {
                dependant.add(new TaskModel(childTaskData, { parse: true }));
            });

            if (prevAttributes && prevAttributes.dependant && prevAttributes.dependant instanceof Backbone.Collection) {
                dependant.each(model => {
                    const prevModel = (
                        prevAttributes.dependant.where({ id: model.get('id') })) || [];

                    if (prevModel && prevModel.length) {
                        const selected = prevModel[0].get('selected');

                        if (selected !== null && selected !== undefined) {
                            model.set('selected', selected);
                        }
                    }
                });
            }

            self.set('dependant', dependant);
        });
    },

    fetchChildren(params) {
        const TasksCollection = require('../collection/TasksCollection');
        const self = this;
        const children = new TasksCollection();
        const prevAttributes = self._previousAttributes;

        return this.fetchAdditionalData('children', params).done(data => {
            data.items = _.sortBy(data.items, child => {
                return (-1 * child.id);
            });
            data.items.forEach(childTaskData => {
                children.add(new TaskModel(childTaskData, { parse: true }));
            });

            if (prevAttributes && prevAttributes.children && prevAttributes.children instanceof Backbone.Collection) {
                children.each(model => {
                    const prevModel = (
                        prevAttributes.children.where({ id: model.get('id') })) || [];

                    if (prevModel && prevModel.length) {
                        const selected = prevModel[0].get('selected');

                        if (selected !== null && selected !== undefined) {
                            model.set('selected', selected);
                        }
                    }
                });
            }

            self.set('children', children);
        });
    },

    fetchUnfilteredChildren() {
        const TasksCollection = require('../collection/TasksCollection');
        const self = this;
        const children = new TasksCollection();

        return $.ajax({
            url: '/api/v1.0/task/?' +
                'limit=100500&' +
                'order=-id&' +
                'hidden=true&' +
                'status=ALL&' +
                'parent=' + this.get('id')
        }).done(data => {
            data.items.forEach(childTaskData => {
                children.add(new TaskModel(childTaskData, { parse: true }));
            });

            self.set('children', children);
        });
    },

    /**
     * @description Will fetch task custom parameters from the server, but if it already fetched,
     *              will map same-named properties values to .context.value props.
     *
     * @param   {String}          mode
     * @returns {jQuery.Deferred}
     */
    fetchCustomParameters(mode) {
        const self = this;
        let customFields = this.get('customFields');

        if (customFields instanceof Array && (!mode || mode === this.get('customFieldsMode'))) {
            customFields = this.mapToCustomFields(customFields);
            this.set('customFields', customFields);

            return $.Deferred().resolve(customFields); // eslint-disable-line
        }
        return this.fetchAdditionalData('custom/fields', {})
            .done(data => {
                self.set('customFields', data);
                self.set('customFieldsMode', mode);
            });
    },

    fetchResources() {
        const self = this;

        return this.fetchAdditionalData('resources').done(data => {
            const ResourcesCollection = require('../collection/TaskResourcesCollection');

            self.set('resources', new ResourcesCollection(data.items));
        });
    },

    fetchReleaseInfo() {
        const fetched = $.Deferred(); // eslint-disable-line
        const ReleaseModel = require('./ReleaseModel');

        if (this.get('status') === TaskModel.STATUS.RELEASED) {
            const self = this;
            const release = new ReleaseModel({ id: this.get('id') });

            release
                .fetch()
                .always(() => {
                    self.set('release', _.extend(
                        {},
                        self.get('release'),
                        release.toJSON()
                    ));
                    self.trigger('change:release');
                })
                .done(() => {
                    fetched.resolve();
                })
                .fail(function () {
                    fetched.reject(arguments[0]);
                });
        } else {
            fetched.resolve();
        }

        return fetched;
    },

    fetchDependOn() {
        const self = this;

        return this.fetchAdditionalData('requirements').done(data => {
            const requirements = self.get('requirements');
            const ResourcesCollection = require('../collection/TaskResourcesCollection');

            requirements.resources = new ResourcesCollection(data.items);

            if (requirements.ramdrive && requirements.ramdrive.type) {
                requirements.ramdrive = requirements.ramdrive.size;
            }

            self.set('requirements', requirements);
        });
    },

    getReportTabs() {
        const reports = this.get('reports');

        return reports && reports.filter(report => {
            return report.label !== 'footer' && report.label !== 'header';
        });
    },

    fetchReport(type) {
        const self = this;
        const reports = this.get('reports');

        const fetchableReport = reports && reports.filter(report => {
            return report.label === type;
        })[0];

        if (!fetchableReport) {
            return $.Deferred().resolve(); // eslint-disable-line
        }

        return $.ajax({
            url: fetchableReport.url,
            type: 'GET',
            dataType: 'json',
            contentType: 'application/json',
            fail: error.fromXHR
        }).then(data => {
            return app.getHelperManager().processDataWithViewHelpers(data).done(data => {
                self.set(fetchableReport.label, data);
            });
        });
    },

    fetchHosts() {
        const self = this;

        return this.fetchAdditionalData('queue', { all: true }).done(data => {
            const HostsCollection = require('../collection/TaskHostsCollection');

            self.set('hosts', new HostsCollection(data));
        });
    },

    /**
     * @returns {*|JQueryDeferred}
     */
    fetchClient() {
        const execInfo = this.get('execution');

        if (!execInfo.client || !execInfo.client.id) {
            throw new Error('SANDBOX ERROR: can not fetch client info for NON-EXECUTING task');
        }

        const task = this;
        const client = new ClientModel({ id: execInfo.client.id, isFromTask: true });

        return client.fetch().done(() => {
            task.set('client', client);
        });
    },

    fetchAdditionalData(dataType, data) {
        return $.ajax({
            url: this.url() + '/' + dataType,
            type: 'GET',
            data: (data ? data : {}),
            dataType: 'json',
            contentType: 'application/json',
            fail: error.fromXHR
        });
    },

    duplicate() {
        const self = this;

        this.set('_duplicating', true);

        return $.ajax({
            url: '/api/v1.0/task',
            type: 'POST',
            data: JSON.stringify({
                source: this.get('id')
            }),
            dataType: 'json',
            contentType: 'application/json'
        }).done(response => {
            self.set('_duplicating', false);

            const newTask = new TaskModel();

            newTask.set(TaskModel.prototype.parse.call(newTask, response));
            app.setDraftTask(newTask);
        })
            .fail(err => {
                self.set('_duplicating', false);

                error.fromXHR(err);
            });
    },

    canBeClonned() {
        return !this.get('_duplicating');
    },

    canBeReleased() {
        const status = this.get('status');

        return (
            this.get('rights').includes('write') &&
            !_.isUndefined(this.get('release')) &&
            (
                status === TaskModel.STATUS.SUCCESS ||
                status === TaskModel.STATUS.RELEASED ||
                status === TaskModel.STATUS.NOT_RELEASED
            )
        );
    },

    canBeReleasedWithMessage() {
        let message = '';

        if (!this.get('rights').includes('write')) {
            message = 'You must be a member of owner group and a releaser of each resource to release task';
        } else if (_.isUndefined(this.get('release'))) {
            message = 'Nothing to release';
        } else {
            message = 'Task can not be released from the current state';
        }

        return {
            canBeReleased: this.canBeReleased(),
            message
        };
    },

    canBeStopped() {
        const status = this.get('status');
        const canBeStoppedFrom = [
            TaskModel.STATUS.ENQUEUED,
            TaskModel.STATUS.ENQUEUING,
            TaskModel.STATUS.EXECUTING,
            TaskModel.STATUS.TEMPORARY,
            TaskModel.STATUS.PREPARING,
            TaskModel.STATUS.ASSIGNED,
            TaskModel.STATUS.SUSPENDED
        ];

        return this.get('rights').includes('write') &&
            (
                canBeStoppedFrom.includes(status) ||
                TaskModel.STATUSES.WAIT.includes(status) ||
                TaskModel.VIRTUAL_STATUSES.WAIT.includes(status)
            );
    },

    canBeSuspended() {
        const status = this.get('status');
        const canBeSuspendedFrom = [
            TaskModel.STATUS.EXECUTING,
            TaskModel.STATUS.PREPARING,
            TaskModel.STATUS.ASSIGNED
        ];

        return (
            this.get('rights').includes('write') &&
            canBeSuspendedFrom.includes(status)
        );
    },

    canBeExecuted() {
        const status = this.get('status');

        return (
            TaskModel.STATUSES.BREAK.includes(status) ||
            TaskModel.STATUSES.DRAFT.includes(status) ||
            TaskModel.STATUSES.PAUSED.includes(status) || status === 'WAIT_TIME'
        );
    },

    canBeResumed() {
        const status = this.get('status');

        return (status === TaskModel.STATUS.SUSPENDED || status === TaskModel.STATUS.SUSPENDING);
    },

    canBeDeleted() {
        const status = this.get('status');
        const canBeDeletedFrom = [
            TaskModel.STATUS.ENQUEUING,
            TaskModel.STATUS.ENQUEUED,
            TaskModel.STATUS.SUCCESS,
            TaskModel.STATUS.NOT_RELEASED,
            TaskModel.STATUS.RELEASING,
            TaskModel.STATUS.RELEASED,
            TaskModel.STATUS.FAILURE,
            TaskModel.STATUS.NO_RES,
            TaskModel.STATUS.EXCEPTION,
            TaskModel.STATUS.TIMEOUT,
            TaskModel.STATUS.EXPIRED,
            TaskModel.STATUS.STOPPED,
            TaskModel.STATUS.SUSPENDED
        ];

        return ((
            TaskModel.STATUSES.WAIT.includes(status) ||
            TaskModel.STATUSES.DRAFT.includes(status) ||
            TaskModel.STATUSES.EXECUTE.includes(status) ||
            canBeDeletedFrom.includes(status)
        ) && this.get('rights') === TaskModel.RIGHTS.WRITE);
    },

    canBeEdited() {
        return (this.get('status') === TaskModel.STATUS.DRAFT && this.get('rights') === TaskModel.RIGHTS.WRITE);
    },

    canPriorityBeIncreased() {
        // Const user = app.request('USER'); @TODO remove fast fix for USER:HIGH priorities
        const status = this.get('status');

        if (status === 'ENQUEUING') {
            return false;
        }

        if (!TaskModel.STATUSES.WAIT.includes(status) &&
            !TaskModel.STATUSES.QUEUE.includes(status)) {
            return false;
        }

        if (this.get('rights') !== TaskModel.RIGHTS.WRITE/* && user.get('role') !== UserModel.ROLES.ADMIN */) {
            // @TODO remove fast fix for USER:HIGH priorities
            return false;
        }

        return true;
    },

    getCustomFieldValue(name) {
        const found = _.where(this.get('customFields'), { name });

        return (found.length ? found[0].value : null);
    },

    setCustomFieldValue(name, value) {
        const found = _.where(this.get('customFields'), { name });

        if (found.length) {
            found[0].value = value;
        }
    },

    offProps() {
        const children = this.get('children');

        if (children && children instanceof Backbone.Collection) {
            children.off();
            children.stopListening();
            children.reset();
            this.set('children', null);
        }

        const dependant = this.get('dependant');

        if (dependant && dependant instanceof Backbone.Collection) {
            dependant.off();
            this.set('dependant', null);
        }
    },

    getChildLevel() {
        if (typeof this.childLevel === 'undefined') {
            this.childLevel = 0;
        }

        return this.childLevel;
    },

    setChildLevel(childLevel) {
        this.childLevel = childLevel;
    }
}, {

        getPossibleStatuses() {
            const statuses = [];

            Object.keys(TaskModel.STATUSES).forEach(groupName => {
                TaskModel.STATUSES[groupName].forEach(status => {
                    statuses.push(status);
                });
            });

            return statuses;
        },

        getPossibleGroups() {
            return Object.keys(TaskModel.STATUSES);
        },

        getStatusGroups() {
            return TaskModel.STATUSES;
        },

        getVirtualStatusGroups() {
            return TaskModel.VIRTUAL_STATUSES;
        },

        getStatusUpdateIntervals() {
            return [
                { key: 'app_period_1minute', label: '1 minute' },
                { key: 'app_period_10minute', label: '10 minutes' },
                { key: 'app_period_30minute', label: '30 minutes' }
            ];
        },

        createBlankTask() {
            const task = new TaskModel();

            delete task.attributes;
            task.attributes = {};

            return task;
        },

        PRIORITY: {

            CLASS: [
                'BACKGROUND',
                'SERVICE',
                'USER'
            ],

            SUBCLASS: [
                'LOW',
                'NORMAL',
                'HIGH'
            ]
        },

        STATUSES: {
            WAIT: ['WAIT_RES', 'WAIT_TASK', 'WAIT_TIME', 'WAIT_MUTEX', 'WAIT_OUT'],
            DRAFT: ['DRAFT'],
            QUEUE: ['ENQUEUING', 'ENQUEUED'],
            BREAK: ['NO_RES', 'STOPPING', 'EXCEPTION', 'TIMEOUT', 'STOPPED', 'EXPIRED'],
            FINISH: ['SUCCESS', 'RELEASING', 'NOT_RELEASED', 'RELEASED', 'FAILURE', 'DELETED'],
            EXECUTE: ['PREPARING', 'ASSIGNED', 'EXECUTING', 'TEMPORARY', 'FINISHING', 'SUSPENDING', 'SUSPENDED'],
            PAUSED: ['TEMPORARY', 'SUSPENDING', 'SUSPENDED']
        },

        VIRTUAL_STATUSES: {
            WAIT: ['WAIT_MUTEX']
        }
    });

/**
 * @memberof TaskModel
 * @static
 */
TaskModel.PROP_SET = {
    time: null,
    type: '',
    owner: '',
    queue: [],
    author: '',
    status: '',
    rights: [],
    scores: -1,
    se_tag: null,
    parent: {},
    se_tags: [],
    reports: [],
    selected: null,
    children: null,
    lock_host: '',
    priority: {
        class: TaskModel.PRIORITY.CLASS[1],
        subclass: TaskModel.PRIORITY.SUBCLASS[1]
    },
    tasks_archive_resource: 0,
    requirements: {
        disk_space: 0,
        platform: '',
        ram: 0
    },
    hidden: false,
    important: false,
    execution: { current: 0, estimated: 0 },
    kill_timeout: '',
    description: '',
    fail_on_any_error: false,
    relatedHosts: []
};

/**
 * @memberof TaskModel
 * @static
 */
TaskModel.STATUS = {
    DRAFT: 'DRAFT',
    DELETED: 'DELETED',
    ENQUEUED: 'ENQUEUED',
    ENQUEUING: 'ENQUEUING',
    EXECUTING: 'EXECUTING',
    EXCEPTION: 'EXCEPTION',
    FAILURE: 'FAILURE',
    FINISHED: 'FINISHED',
    FINISHING: 'FINISHING',
    NOT_READY: 'NOT_READY',
    RELEASED: 'RELEASED',
    RELEASING: 'RELEASING',
    NOT_RELEASED: 'NOT_RELEASED',
    STOPPED: 'STOPPED',
    STOPPING: 'STOPPING',
    UNKNOWN: 'UNKNOWN',
    WAIT_CHILD: 'WAIT_CHILD',
    WAIT_DEPS: 'WAIT_DEPS',
    WAIT_MUTEX: 'WAIT_MUTEX',
    SUCCESS: 'SUCCESS',
    NO_RES: 'NO_RES',
    SUSPENDED: 'SUSPENDED',
    SUSPENDING: 'SUSPENDING',
    TIMEOUT: 'TIMEOUT',
    EXPIRED: 'EXPIRED',
    PREPARING: 'PREPARING',
    TEMPORARY: 'TEMPORARY',
    ASSIGNED: 'ASSIGNED'
};

/**
 * @memberof TaskModel
 * @static
 */
TaskModel.CUSTOM_PAR_MODE = {
    VIEW: 'view',
    EDIT: 'edit'
};

/**
 * @memberof TaskModel
 * @static
 */
TaskModel.RIGHTS = {
    WRITE: 'write',
    READ: 'read'
};

module.exports = TaskModel;
