/* eslint camelcase: 0 */

const TasksCollection = require('../collection/TasksCollection');

/**
 * @class TaskResourceModel
 * @extends Backbone.Model
 */
const TaskResourceModel = Backbone.Model.extend({

    url() {
        const baseURL = '/api/v1.0/resource';

        return (this.get('id') ? (baseURL + '/' + this.get('id')) : baseURL);
    },

    serializeAttrs() {
        let attrs = [];
        const values = this.get('attributes');

        Object.keys(values).forEach(key => {
            attrs.push({
                name: key,
                value: values[key]
            });
        }, this);

        if (attrs.length > 1) {
            attrs = _.sortBy(attrs, 'name');
        }

        return attrs;
    },

    defaults: {
        skynet_id: '',
        rsync: {
            links: []
        },
        task: {
            url: '',
            status: '',
            id: 0
        },
        http: {
            links: [],
            proxy: ''
        },
        attributes: {},
        description: '',
        file_name: '',
        rights: 'write',
        state: '',
        arch: '',
        time: {
            accessed: '',
            created: '',
            expires: ''
        },
        selected: null,
        dependant: null,
        owner: '',
        size: 0,
        type: '',
        md5: '',
        id: 0
    },

    initialize() {
        this.listenTo(this, 'actionSuccess', this.fetch);

        this.fetchDependant = _.memoize(this.fetchDependant);
    },

    set(attrs, options) {
        if (attrs.selected === null) {
            if (this.attributes.selected !== null) {
                attrs.selected = this.attributes.selected;
            }
        }

        Backbone.Model.prototype.set.call(this, attrs, options);
    },

    setSelected(isSelected, options) {
        options = _.extend({ silent: false, isBatch: false }, options);

        this.set({
            selected: (typeof isSelected === 'undefined' ? !this.get('selected') : Boolean(isSelected))
        }, {
            silent: Boolean(options.silent),
            isBatch: Boolean(options.isBatch)
        });
    },

    toggleSelected(options) {
        this.setSelected(!this.get('selected'), options);
    },

    canBeRestored() {
        return (
            (
                this.get('state') === TaskResourceModel.STATE.BROKEN ||
                this.get('state') === TaskResourceModel.STATE.DELETED
            ) &&
            this.get('rights') === TaskResourceModel.RIGHTS.WRITE
        );
    },

    canBeChanged() {
        return (
            this.get('state') === TaskResourceModel.STATE.READY &&
            this.get('rights') === TaskResourceModel.RIGHTS.WRITE
        );
    },

    fetchDependant() {
        const self = this;
        const dependant = new TasksCollection([], {
            requires: this.get('id'),
            children: true,
            hidden: true
        });

        return dependant.update().done(() => {
            self.set('dependant', dependant);
        });
    },

    getDependantQnt() {
        const dependant = this.get('dependant');

        return (dependant ? dependant.length : 0);
    }
}, {
    getPossibleStatuses() {
        return _.values(TaskResourceModel.STATE);
    },

    RIGHTS: {
        WRITE: 'write'
    },

    STATE: {
        READY: 'READY',
        BROKEN: 'BROKEN',
        DELETED: 'DELETED',
        NOT_READY: 'NOT_READY'
    }
});

module.exports = TaskResourceModel;
