/* eslint camelcase: 0 */

const sanitize = require('sanitize-html');
const error = require('../../error');

/**
 * @class ClientModel
 * @extends Backbone.Model
 */
const ClientModel = Backbone.Model.extend({

    url() {
        return ('/api/v1.0/client/' + this.get('id'));
    },

    defaults: {
        // Utility attributes
        selected: null,

        // Content attributes
        host_name: '',
        venv: 0,
        ram: '',
        alive: false,
        ncpu: 0,
        tags: '',
        tasks: 0,
        configs: 0,
        disk: {
            status: '',
            total_space: 0,
            free_space: 0
        },
        id: 0,
        sdk: 0,
        task: null,
        url: '',
        server: 0,
        platform: '',
        platforms: [],
        last_activity: null,
        version: 0,
        os: {
            version: '',
            name: ''
        },
        cpu: '',
        msg: ''
    },

    initialize() {
        this.listenTo(this, 'change:msg', this.onMsgUpdate);
    },

    parse(data) {
        if (data.msg) {
            data.msg = sanitize(data.msg, { allowedTags: [] });
        }

        if (_.isNull(data.platforms)) {
            data.platforms = [];
        }

        if (data.last_activity) {
            data.last_activity = moment(data.last_activity);
        }

        return data;
    },

    getGroupableProps() {
        const task = this.get('task') || {
            owner: '',
            url: '',
            task_type: '',
            task_id: 0
        };
        const platform = this.get('platforms');

        return {
            dc: this.get('dc'),
            os: ((!platform || !platform.length) ? this.get('platform') : platform),
            ram: this.get('ram'),
            cpu: this.get('cpu'),
            ncpu: this.get('ncpu'),
            type: task.task_type,
            owner: task.owner
        };
    },

    onMsgUpdate() {
        const self = this;

        if (self.get('isFromTask')) {
            return;
        }

        return $.ajax({
            url: '/api/v1.0/client/' + this.get('id') + '/comment',
            type: 'PUT',
            data: this.get('msg'),
            dataType: 'json',
            contentType: 'application/json'
        })
            .fail(xhr => {
                self.trigger('msgUpdateFailed');
                error.fromXHR(xhr);
            });
    },

    serializeData() {
        const serialized = this.toJSON();

        if (serialized.last_activity) {
            serialized.last_activity = serialized.last_activity.format('DD.MM.YYYY HH:mm:ss');
        }

        return serialized;
    }
}, {

    fetchQueue(hostId) {
        const fetched = $.Deferred(); // eslint-disable-line

        if (!hostId) {
            fetched.reject();
        }

        $.ajax({
            url: '/api/v1.0/service/q',
            type: 'GET',
            data: {
                client: hostId,
                limit: 100500
            },
            dataType: 'json',
            contentType: 'application/json'
        }).done(queueInfo => {
            fetched.resolve(queueInfo);
        }).fail((xhr, state, error) => {
            fetched.reject(xhr, state, error);
        });

        return fetched;
    }

});

module.exports = ClientModel;
