const router = require('../../router');

/**
 * @description Basic functionality of the filter/preset models.
 *
 * @class BasicFilterModel
 * @extends Backbone.Model
 */
const BasicFilterModel = Backbone.Model.extend({

    filterItemName: '',

    idAttribute: 'order',

    defaults: {
        title: '',
        found: 0,
        order: -1
    },

    dropSelfAttrs(instance) {
        delete instance.title;
        delete instance.order;
        delete instance.found;
    },

    setPresets(presetedParams) {
        this.set(this.parse(_.extend({}, this.defaults, presetedParams)));
    },

    setOrder(order) {
        order = parseInt(order, 10);

        if (order && order > 0 && this.get('order') !== order) {
            this.set({ order });
        }
    },

    setTitle(titleToSet) {
        this.set('title', titleToSet);
    },

    /**
     * @description Will drop self attributes and empty values to NOT show them in the URL.
     * @returns {Object} Hash of non-empty objects.
     */
    getFilterParams() {
        const filterParams = this.toJSON();
        const filtered = {};

        this.dropSelfAttrs(filterParams);

        Object.keys(filterParams).forEach(key => {
            if (filterParams[key] !== '' && filterParams[key] !== null) {
                filtered[key] = filterParams[key];
            }
        });

        return filtered;
    },

    /**
     * @description Serializes the model for making the query.
     * @returns {Object}
     */
    serializeAsQuery() {
        const serialized = _.clone(this.attributes);

        delete serialized.order;

        return router.buildQueryJson(serialized);
    },

    /**
     * @description Serializes the model to be mirrored in the URL.
     *              All empty values will be dropped.
     *
     * @returns {Object}
     */
    serializeAsUrl() {
        const serialized = this.getFilterParams();

        delete serialized.order;
        delete serialized.fields;

        return router.buildQueryJson(serialized);
    },

    /**
     * @description Makes a request to get a number of items that are match a preset.
     * @returns {JQueryXHR}
     */
    getFound() {
        const self = this;

        return $.get('/api/v1.0/' +
            this.filterItemName +
            '?limit=0&' +
            'offset=0&' +
            router.buildQueryString(this.serializeAsQuery()), response => {
            if (response && response.total) {
                self.set('found', response.total);
            }
        });
    },

    saveFilterState() {
        return $.ajax({
            url: this.url,
            type: 'PUT',
            data: JSON.stringify(this.toJSON()),
            dataType: 'json',
            contentType: 'application/json'
        });
    }
}, {

    /**
     * @description Tests if some hash of attributes match any keys with default filter dataset.
     *
     * @param {Object} defaultValues - hash of defaults
     * @param {Object} hashToCheck   - hash of values to check
     *
     * @returns {boolean}
     */
    hasFilterValues(defaultValues, hashToCheck) {
        const checkKeys = Object.keys(hashToCheck);
        const defaultKeys = Object.keys(defaultValues);

        let result = _.some(checkKeys, item => {
            return _.contains(defaultKeys, item);
        });

        if (!result && hashToCheck && hashToCheck.hasOwnProperty('order')) { // eslint-disable-line
            result = true;
        }

        return result;
    }
});

module.exports = BasicFilterModel;
