/* eslint camelcase: 0 */
/* eslint complexity: [1, 14] */
/* eslint max-statements: [1, 22] */

const DateTimeIntervalModel = require('./../DateTimeIntervalModel');
const BasicFilterModel = require('./BasicFilterModel');
// eslint-disable-next-line no-unused-vars
const router = require('../../router');
// eslint-disable-next-line no-unused-vars
const error = require('../../error');

/**
 * @description Filter model for the tasks lists. And it's presets.
 *
 * @class FilterModel
 * @extends BasicFilterModel
 *
 * @property {String} filterItemName - Need to build preset capacity fetch URL
 * @property {String} url - Defined to match the last saved state of the filter for the tasks list
 */
const FilterModel = BasicFilterModel.extend({

    filterItemName: 'task',

    url: '/api/v1.0/user/current/preferences/task-list-last-filter',

    defaults() {
        return _.extend(
            {},
            BasicFilterModel.prototype.defaults,
            FilterModel.getDefaults()
        );
    },

    options: {},

    initialize(data, options) {
        this.set(this.parse(data));
        this.options = _.extend({}, this.options, options);
    },

    toJSON() {
        const jsonfied = _.extend(_.clone(this.attributes), {
            status: typeof this.get('status') === 'string' ?
                this.get('status').toUpperCase() :
                this.get('status').map(item => item.toUpperCase()).join(',')
        });

        if (this.get('created')) {
            jsonfied.created = this.get('created').toJSON();
        }

        if (this.get('updated')) {
            jsonfied.updated = this.get('updated').toJSON();
        }

        return jsonfied;
    },

    /**
     * @override
     */
    getFilterParams() {
        const filterParams = this.toJSON();
        const filtered = {};

        this.dropSelfAttrs(filterParams);

        Object.keys(filterParams).forEach(key => {
            if (key === 'created' || key === 'updated') {
                if (filterParams[key].offset) {
                    filtered[key] = filterParams[key];
                }
            } else if (filterParams[key] !== '' && filterParams[key] !== null) {
                filtered[key] = filterParams[key];
            }
        });

        return filtered;
    },

    parse(data) {
        if (data.created && !(data.created instanceof DateTimeIntervalModel)) {
            data.created = new DateTimeIntervalModel(DateTimeIntervalModel.prototype.parse.call(null, data.created));
        }

        if (data.updated && !(data.updated instanceof DateTimeIntervalModel)) {
            data.updated = new DateTimeIntervalModel(DateTimeIntervalModel.prototype.parse.call(null, data.updated));
        }

        if (typeof data.status === 'string') {
            data.status = (data.status ? data.status.split(',').map(item => item.toUpperCase()) : []);
        }

        if (typeof data.input_parameters === 'string') {
            try {
                data.input_parameters = JSON.parse(data.input_parameters);
            } catch (e) {}
        }

        if (typeof data.output_parameters === 'string') {
            try {
                data.output_parameters = JSON.parse(data.output_parameters);
            } catch (e) {}
        }

        if (typeof data.any_params === 'string') {
            data.any_params = data.any_params === 'false' ?
                false : data.any_params === 'true' ?
                    true : Boolean(data.any_params);
        }

        return data;
    },

    resetFilter() {
        this.set(FilterModel.getDefaults());
        this.trigger('change:created', this, this.get('created'));
        this.trigger('change:updated', this, this.get('updated'));
        this.trigger('blank', this);
    },

    serializeAsQuery() {
        const serialized = BasicFilterModel.prototype.serializeAsQuery.apply(this, arguments);

        if (serialized.attrs) {
            serialized.attrs = _.isString(serialized.attrs) ?
                serialized.attrs :
                JSON.stringify(serialized.attrs);
        }

        if (serialized.created) {
            serialized.created = DateTimeIntervalModel.stringify(serialized.created.toJSON());
        }

        if (serialized.updated) {
            serialized.updated = DateTimeIntervalModel.stringify(serialized.updated.toJSON());
        }

        if (serialized.status) {
            serialized.status = (serialized.status.length > 0 ? serialized.status.join(',') : '');
        }

        if (serialized.task_id) {
            serialized.id = serialized.task_id;
            delete serialized.task_id;
        }

        if (serialized.tags instanceof Array) {
            serialized.tags = (serialized.tags.length > 0 ? serialized.tags.join(',') : '');
        }

        if (serialized.hints instanceof Array) {
            serialized.hints = (serialized.hints.length > 0 ? serialized.hints.join(',') : '');
        }

        if (serialized.input_parameters) {
            serialized.input_parameters = JSON.stringify(serialized.input_parameters);
        }

        if (serialized.output_parameters) {
            serialized.output_parameters = JSON.stringify(serialized.output_parameters);
        }

        return router.buildQueryJson(serialized);
    },

    serializeAsUrl() {
        const serialized = BasicFilterModel.prototype.serializeAsUrl.apply(this, arguments);

        if (serialized.attrs) {
            serialized.attrs = _.isString(serialized.attrs) ?
                serialized.attrs :
                JSON.stringify(serialized.attrs);
        }

        if (serialized.created) {
            serialized.created = DateTimeIntervalModel.humanify(serialized.created);
        }

        if (serialized.updated) {
            serialized.updated = DateTimeIntervalModel.humanify(serialized.updated);
        }

        if (serialized.tags instanceof Array) {
            serialized.tags = (serialized.tags.length > 0 ? serialized.tags.join(',') : '');
        }

        if (serialized.hints instanceof Array) {
            serialized.hints = (serialized.hints.length > 0 ? serialized.hints.join(',') : '');
        }

        if (serialized.input_parameters) {
            serialized.input_parameters = JSON.stringify(serialized.input_parameters);
        }

        if (serialized.output_parameters) {
            serialized.output_parameters = JSON.stringify(serialized.output_parameters);
        }

        return router.buildQueryJson(serialized);
    }
}, {

    parseOuterFilterValues(outerValues) {
        const parsed = {};

        Object.keys(FilterModel.getDefaults()).forEach(key => {
            try {
                if (typeof outerValues[key] !== 'undefined') {
                    let value = outerValues[key];

                    if (key === 'created' || key === 'updated') {
                        if (typeof outerValues[key] === 'string') {
                            /* eslint max-depth: [1, 4] */

                            value = DateTimeIntervalModel.parseStringifiedData(value);
                        }
                    }

                    if (key === 'status') {
                        value = value.toUpperCase();
                    }

                    parsed[key] = value;
                }
            } catch (excpt) {
                /* eslint no-console: 0 */
                console.warn('SANDBOX WARN: problem with parsing «' + key + '» filter value.');
            }
        });

        return parsed;
    },

    hasFilterValues(hashToCheck) {
        return BasicFilterModel.hasFilterValues(FilterModel.getDefaults(), hashToCheck);
    },

    getDefaults() {
        return {
            id: '',
            task_id: '',
            tags: [],
            all_tags: false,
            hints: [],
            all_hints: false,
            host: '',
            arch: '',
            type: '',
            owner: '',
            model: '',
            hidden: null,
            author: '',
            status: [],
            children: null,
            important: null,
            desc_re: '',
            scheduler: '',
            resource_id: '',
            created: new DateTimeIntervalModel({}, { dateOnly: true }),
            updated: new DateTimeIntervalModel(),
            requires: '',
            forPage: 'tasks',
            input_parameters: null,
            output_parameters: null,
            any_params: null,
            parent: null,
            priority: null,
            se_tag: null,
            fields: null
        };
    }
});

module.exports = FilterModel;
