/* eslint camelcase: 0 */

const router = require('../../router');
const BasicFilterModel = require('./BasicFilterModel');
const DateTimeIntervalModel = require('./../DateTimeIntervalModel');

/**
 * @description Filter model for the resources lists (resources page and modal window on the task create/edit page).
 *              And it's presets.
 *
 * @class ResourceFilterModel
 * @extends BasicFilterModel
 *
 * @property {String} filterItemName - Need to build preset capacity fetch URL
 * @property {String} url - Defined to match the last saved state of the filter for the resources list
 */
const ResourceFilterModel = BasicFilterModel.extend({

    filterItemName: 'resource',

    url: '/api/v1.0/user/current/preferences/resource-list-last-filter',

    options: {},

    defaults() {
        return _.clone(ResourceFilterModel.getDefaults());
    },

    initialize(data, options) {
        this.set(this.parse(data));
        this.options = _.extend({}, this.options, options);
    },

    toJSON() {
        const jsonfield = _.extend({}, this.attributes, {
            hidden: (this.get('hidden') ? 1 : 0)
        });

        if (this.get('created')) {
            jsonfield.created = this.get('created').toJSON();
        }

        return jsonfield;
    },

    getFilterParams() {
        const filterParams = this.toJSON();
        const filtered = {};

        this.dropSelfAttrs(filterParams);

        Object.keys(filterParams).forEach(key => {
            if (key === 'created') {
                if (filterParams[key].offset) {
                    filtered[key] = filterParams[key];
                }
            } else if (filterParams[key] !== '' && filterParams[key] !== null) {
                filtered[key] = filterParams[key];
            }
        });

        return filtered;
    },

    resetFilter() {
        this.set(ResourceFilterModel.getDefaults());
        this.trigger('change:created', this, this.get('created'));
        this.trigger('blank', this);
    },

    parse(data) {
        if (typeof data.any_params === 'string') {
            data.any_params = data.any_params === 'false' ?
                false : data.any_params === 'true' ?
                    true : Boolean(data.any_params);
        }

        if (data.created && !(data.created instanceof DateTimeIntervalModel)) {
            data.created = new DateTimeIntervalModel(DateTimeIntervalModel.prototype.parse.call(null, data.created));
        }

        if (typeof data.attrs === 'string') {
            try {
                data.attrs = JSON.parse(data.attrs);
            } catch (e) {}
        }

        return data;
    },

    serializeAsUrl() {
        const serialized = BasicFilterModel.prototype.serializeAsUrl.apply(this, arguments);

        if (serialized.created) {
            serialized.created = DateTimeIntervalModel.humanify(serialized.created);
        }

        if (serialized.attrs) {
            serialized.attrs = JSON.stringify(serialized.attrs);
        }

        if (serialized.tags instanceof Array) {
            serialized.tags = (serialized.tags.length > 0 ? serialized.tags.join(',') : '');
        }

        return router.buildQueryJson(serialized);
    },

    serializeAsQuery() {
        const serialized = BasicFilterModel.prototype.serializeAsQuery.apply(this, arguments);

        if (serialized.created) {
            serialized.created = DateTimeIntervalModel.stringify(serialized.created.toJSON());
        }

        if (serialized.attrs) {
            serialized.attrs = JSON.stringify(serialized.attrs);
        }

        if (serialized.tags instanceof Array) {
            serialized.tags = (serialized.tags.length > 0 ? serialized.tags.join(',') : '');
        }

        return router.buildQueryJson(serialized);
    }

}, {

    parseOuterFilterValues(outerValues) {
        const parsed = {};

        Object.keys(ResourceFilterModel.getDefaults()).forEach(key => {
            try {
                if (typeof outerValues[key] !== 'undefined') {
                    let value = outerValues[key];

                    if (key === 'created') {
                        if (typeof outerValues[key] === 'string') {
                            value = DateTimeIntervalModel.parseStringifiedData(value);
                        }
                    }

                    parsed[key] = value;
                }
            } catch (excpt) {
                /* eslint no-console: 0 */
                console.warn('SANDBOX WARN: problem with parsing «' + key + '» filter value.');
            }
        });

        return parsed;
    },

    hasFilterValues(hashToCheck) {
        return BasicFilterModel.hasFilterValues(ResourceFilterModel.getDefaults(), hashToCheck);
    },

    getDefaults() {
        return {
            id: '',
            arch: '',
            type: '',
            any_params: null,
            tags: [],
            all_tags: false,
            owner: '',
            state: '',
            client: '',
            author: '',
            hidden: false,
            task_id: '',
            depend_by: '',
            attr_name: '',
            attr_value: '',
            desc_re: '',
            created: new DateTimeIntervalModel({}, { dateOnly: false })
        };
    }
});

module.exports = ResourceFilterModel;
