const ClientModel = require('../clients/ClientsModel');

/**
 * @class TaskTypeModel
 * @extends Backbone.Model
 */
const TaskTypeModel = Backbone.Model.extend({

    defaults: {
        type: '',
        hosts: null,
        owners: [],
        shortened: '',
        description: '',

        boundHosts: null,
        defaultHosts: null,

        shortenedBoundHosts: '',
        shortenedDefaultHosts: ''
    },

    constructor(attributes) {
        this.id = attributes.id;
        this.cid = _.uniqueId('c');
        this.attributes = attributes;
        this.initialize(attributes);
    },

    initialize() {
        this.fetchBoundHosts = _.memoize(this.fetchBoundHosts);
        this.fetchDefaultHosts = _.memoize(this.fetchDefaultHosts);
        this.fetchShortenedHosts = _.memoize(this.fetchShortenedHosts);
    },

    /**
     * @description Will shortify the list of hosts using server API.
     *
     * @param {Array|String} toShortify
     * @returns {jQuery.Deferred}
     */
    fetchShortenedHosts(toShortify) {
        return $.ajax({
            url: '/api/v1.0/service/shortify/client',
            type: 'POST',
            data: (_.isArray(toShortify) ? JSON.stringify(toShortify) : toShortify),
            dataType: 'text'
        });
    },

    fetchBoundHosts() {
        const self = this;

        return $.ajax({
            url: ('/api/v1.0/service/binding/' + this.get('type')),
            type: 'GET',
            dataType: 'json',
            contentType: 'application/json'
        }).done(boundHosts => {
            self.set('boundHosts', boundHosts, { silent: true });
        });
    },

    fetchDefaultHosts() {
        const self = this;
        const hosts = new ClientModel({}, { type: this.get('type') });

        return hosts
            .fetch()
            .done(client => {
                self.set('defaultHosts', client.get('hosts'), { silent: true });
            });
    },

    loadHostsInfo() {
        const self = this;

        return $.when(this.fetchBoundHosts(), this.fetchDefaultHosts()).then(() => {
            return $.when(
                self.fetchShortenedHosts(self.get('boundHosts')),
                self.fetchShortenedHosts(self.get('defaultHosts'))
            )
                .done((shortenedBoundHosts, shortenedDefaultHosts) => {
                    self.set('shortenedBoundHosts', shortenedBoundHosts[0], { silent: true });
                    self.set('shortenedDefaultHosts', shortenedDefaultHosts[0]);
                });
        });
    },

    getHosts(hostsType) {
        if (hostsType === TaskTypeModel.HOSTS_TYPE.BOUND) {
            return this.get('boundHosts');
        }
        if (hostsType === TaskTypeModel.HOSTS_TYPE.DEFAULT) {
            return this.get('defaultHosts');
        }
        return '';
    },

    getShortenedHosts(hostsType) {
        if (hostsType === TaskTypeModel.HOSTS_TYPE.BOUND) {
            return this.get('shortenedBoundHosts');
        }
        if (hostsType === TaskTypeModel.HOSTS_TYPE.DEFAULT) {
            return this.get('shortenedDefaultHosts');
        }
        return '';
    }
}, {
    HOSTS_TYPE: {
        BOUND: 'bound',
        DEFAULT: 'default'
    }
});

module.exports = TaskTypeModel;
