// eslint-disable-next-line no-unused-vars
const error = require('../../error');

/**
 * @class UserGroupModel
 * @extends Backbone.Model
 */
const UserGroupModel = Backbone.Model.extend({

    url() {
        return ('/api/v1.0/group/' + this.get('name'));
    },

    defaults: {
        name: '',
        sync: null,
        email: '',
        rights: '',
        members: null
    },

    idAttribute: 'name',

    parse(rawGroupModel) {
        if (typeof rawGroupModel === 'string') {
            return {
                name: rawGroupModel
            };
        }
        if (this && this.attributes && this.attributes.name && _.isUndefined(rawGroupModel)) {
            return this.attributes;
        }
        if (rawGroupModel.members) {
            rawGroupModel.members = new Backbone.Collection(_.map(rawGroupModel.members, member => {
                return (_.isString(member) ? { name: member } : member);
            }));
        }

        return rawGroupModel;
    },

    toJSON() {
        const sync = this.get('sync');
        const members = this.get('members');

        return {
            name: this.get('name'),
            sync: (sync ? sync : {
                source: '',
                group: ''
            }),
            email: this.get('email'),
            rights: this.get('rights'),
            members: (members ? members.map(member => {
                return member.get('name');
            }) : [])
        };
    }
}, {

    /**
     * User groups validator
     *
     * @param {string} groupToValidate
     * @returns {jQuery.Deferred}
     */
    validateUserGroup(groupToValidate) {
        const isValid = $.Deferred(); // eslint-disable-line
        const result = {
            toValid: groupToValidate,
            isValid: false
        };

        if (groupToValidate) {
            const userGroup = new UserGroupModel({ name: groupToValidate });

            userGroup
                .fetch()
                .done(() => {
                    result.isValid = true;
                    isValid.resolve(result);
                })
                .fail(() => {
                    isValid.reject(result);
                });
        } else {
            isValid.reject(result);
        }

        return isValid;
    }
});

module.exports = UserGroupModel;
