/**
 * @class HelperManager
 * @extends Backbone.Model
 */
const HelperManager = Backbone.Model.extend({

    defaults: {
        view: new Backbone.Model(),
        edit: new Backbone.Model(),
        validate: new Backbone.Model()
    },

    /**
     * @description Registering helper of some type in the system
     *
     * @param {String}   type
     * @param {String}   name
     * @param {Function} func
     * @returns {boolean}
     */
    registerHelper(type, name, func) {
        let result = false;

        if (typeof name === 'string' && typeof func === 'function') {
            this.get(type).set(name, func);
            result = true;
        }

        return result;
    },

    registerViewHelper(name, func) {
        this.registerHelper(HelperManager.TYPE.VIEW, name, func);
    },

    registerEditHelper(name, func) {
        this.registerHelper(HelperManager.TYPE.EDIT, name, func);
    },

    registerValidationHelper(name, func) {
        this.registerHelper(HelperManager.TYPE.VALIDATE, name, func);
    },

    /**
     * @param {String} type - Type of the helper to get
     * @param {String} name - Name of the helper to get
     *
     * @returns {undefined|Function} Will return helper that will be used for custom fields
     *                               validation or custom footer data preprocessors. Each
     *                               helper should return the $.Deferred
     */
    getHelper(type, name) {
        return this.get(type).get(name);
    },

    getViewHelper(name) {
        return this.getHelper(HelperManager.TYPE.VIEW, name);
    },

    getEditHelper(name) {
        return this.getHelper(HelperManager.TYPE.EDIT, name);
    },

    getValidationHelper(name) {
        return this.getHelper(HelperManager.TYPE.VALIDATE, name);
    },

    processDataWithViewHelpers(data) {
        const self = this;

        data = data || [];

        return this.loadExternalHelpers().then(() => {
            const helpersStack = [];
            const processedData = [];

            data.forEach(dataItem => {
                let helper = null;

                if (dataItem.helperName) {
                    helper = self.getViewHelper(dataItem.helperName);
                }

                helpersStack.push($.when((helper && typeof helper === 'function') ?
                    helper(dataItem.content) :
                    dataItem.content));
            });

            if (helpersStack.length) {
                return jQuery.when
                    .apply(null, helpersStack)
                    .then(function () {
                        _.forEach(arguments, data => {
                            processedData.push(data);
                        });

                        return ({
                            processedData
                        });
                    });
            }

            return ({
                processedData: []
            });
        });
    },

    loadExternalHelpers() {
        const resources = require('../../resources/index.js');

        return resources.externalHelpers();
    }

}, {
    TYPE: {
        VIEW: 'view',
        EDIT: 'edit',
        VALIDATE: 'validate'
    }
});

module.exports = HelperManager;
