const cookie = require('bemmy/src/blocks/component/cookie');
const UserPresetsCollection = require('../collection/user/UserPresetsCollection');
const UserGroupsCollection = require('../collection/resources/UserGroupsCollection');
const UserGroupModel = require('../model/resources/UserGroupModel');

/**
 * @class UserModel
 * @extends Backbone.Model
 */
const UserModel = Backbone.Model.extend({

    url: '/api/v1.0/user',

    defaults() {
        return {
            id: -1,
            uid: 0,
            name: '',
            role: '',
            login: '',
            groups: new UserGroupsCollection(),
            presets: new UserPresetsCollection()
        };
    },

    initialize() {
        this.set('login', cookie.getYandexLogin());
        this.set('name', cookie.getYandexLogin());
        this.set('uid', cookie.getYandexUID());
    },

    setPresets(presets) {
        this.getPresets().stopListening();
        this.set('presets', presets);
    },

    getGroups() {
        return this.get('groups');
    },

    getPresets() {
        return this.get('presets');
    },

    getPreferences() {
        return this.get('preferences');
    },

    isAdmin() {
        return (this.get('role') === UserModel.ROLES.ADMIN);
    },

    /**
     * @returns {JQueryDeferred}
     */
    fetchRole() {
        const self = this;

        return $.ajax({
            url: (this.url + '/current'),
            type: 'GET',
            dataType: 'json',
            contentType: 'application/json'
        }).done(response => {
            if (response) {
                const login = response.login;

                if (login) {
                    self.set('login', login);
                    require('../app').trackMetricTarget('LOGGED_USER', { login });
                    require('../app').trackNotifications({ login });
                }

                if (response.role) {
                    self.set('role', response.role);
                }
            }
        });
    },

    /**
     * @returns {JQueryDeferred}
     */
    fetchGroups() {
        const self = this;

        return $.ajax({
            url: (this.url + '/current/groups'),
            type: 'GET',
            dataType: 'json',
            contentType: 'application/json'
        }).done(response => {
            self.getGroups().set(response.map(function (group) {
                return UserGroupModel.prototype.parse.call(this, group);
            }));
        });
    }
}, {
    ROLES: {
        ADMIN: 'administrator',
        REGULAR: 'regular'
    },

    /**
     * User login validator
     *
     * @param {string} loginToValidate
     * @returns {jQuery.Deferred}
     */
    validateUser(loginToValidate) {
        const isValid = $.Deferred(); // eslint-disable-line
        const result = {
            toValid: loginToValidate,
            isValid: false
        };

        if (loginToValidate) {
            $.ajax({
                url: 'https://center.yandex-team.ru/api/autocomplete/multi/',
                dataType: 'jsonp',
                data: {
                    q: loginToValidate,
                    types: 'staff',
                    limit: 1
                }
            })
                .done(responseArgs => {
                    const response = (responseArgs[0] || {});

                    if (response && response.login === loginToValidate) {
                        result.isValid = true;
                        isValid.resolve(result);
                    }

                    isValid.reject(result);
                })
                .fail(() => {
                    isValid.reject(result);
                });
        } else {
            isValid.reject(result);
        }

        return isValid;
    }
});

module.exports = UserModel;
