/* eslint max-statements: [1, 27] */

const app = require('../app');

const User = require('./../model/user');
const Task = require('./../model/TaskModel');
const router = require('../router');
const Tasks = require('./../collection/TasksCollection');
const Clients = require('./../model/clients/ClientsModel');
const Release = require('../model/ReleaseModel');
const Releases = require('../collection/ReleasesCollection');
const Tags = require('../collection/TagsCollection');
const Dns = require('../collection/DnsCollection');
const Resource = require('./../model/TaskResourceModel');
const TaskTypes = require('./../collection/resources/TaskTypesCollection');
const UserGroups = require('./../collection/resources/UserGroupsCollection');
const ClientsData = require('./../collection/ClientsCollection');
const UserPresets = require('./../collection/user/UserPresetsCollection');
const ResourceTypes = require('./../collection/resources/ResourceTypesCollection');
const UserPreferences = require('./../model/UserPreferences');
const CoreApplication = require('./../application/CoreApplication');
const ShardCollection = require('../collection/ShardCollection');
const SchedulerModel = require('../model/SchedulerModel');
const SchedulersCollection = require('../collection/SchedulersCollection');
const EnqueuedTasksCollection = require('../collection/EnqueuedTasksCollection');
const ServiceThreadCollection = require('../collection/ServiceThreadCollection');
const TaskResourcesCollection = require('./../collection/TaskResourcesCollection');
const VaultsCollection = require('../collection/VaultsCollection');
const SemaphoresCollection = require('../collection/SemaphoresCollection');
const SandboxNotificationsCollection = require('../collection/SandboxNotificationsCollection');
const ServiceNotificationsCollection = require('../collection/ServiceNotificationsCollection');

const resources = {

    taskPropsToLoad: [
        { key: 'updateIntervals', method: 'statusUpdateIntervals' },
        { key: 'resourceTypes', method: 'resourceTypes' },
        { key: 'architectures', method: 'architectures' },
        { key: 'platforms', method: 'platforms' },
        { key: '_tags', method: 'tags' },
        { key: 'cpuModels', method: 'models' },
        { key: 'priorities', method: 'priorities' },
        { key: 'statuses', method: 'statuses' },
        { key: 'groups', method: 'groups' },
        { key: 'hosts', method: 'hosts' },
        { key: 'dns', method: 'dns' },
        { key: 'types', method: 'types' }
    ],

    /**
     * @returns {JQueryDeferred}
     */
    user(options, resourceLocator) {
        const app = require('../app');
        const user = new User();

        return $.when(
            user.fetchRole(),
            user.fetchGroups(),
            this.presets((options && options.presets) ?
                options.presets :
                {}),
            this.userPreferences()
        ).then((role, groups, presets, prefs) => {
            user.set({ preferences: prefs });
            user.setPresets(presets);

            if (resourceLocator) {
                resourceLocator.setUser(user);
            }

            app.request('USER', user);

            return user;
        });
    },

    presets(options) {
        const fetched = $.Deferred(); // eslint-disable-line
        const presets = new UserPresets([], options);

        presets.update().always(() => {
            fetched.resolve(presets);
        });

        return fetched;
    },

    userPreferences() {
        return this.fetchFailable(UserPreferences);
    },

    /**
     * @description Tests if the URL has any params matching current pages' filter.
     *              Returns fetched filter model.
     *
     * @param {Object} options
     * @param {BasicFilterModel} options.filterModel
     *
     * @returns {JQueryDeferred}
     */
    userLastFilter(options) {
        const result = $.Deferred(); // eslint-disable-line
        const locationSearch = window.location.search;
        const querySearchValues = locationSearch && router.parseQueryString(locationSearch);
        const hasFilterQueryString = querySearchValues && options.filterModel.hasFilterValues(querySearchValues);
        let filterModel;

        if (hasFilterQueryString) {
            const FModel = options.filterModel;

            filterModel = new FModel(querySearchValues);

            filterModel.saveFilterState();

            result.resolve(filterModel);
        } else {
            const FModel = options.filterModel;

            filterModel = new FModel({});

            filterModel.fetch({ reset: true }).always(() => {
                result.resolve(filterModel);
            });
        }

        return result;
    },

    /**
     * @param options
     * @returns {JQueryDeferred}
     */
    tasks(options) {
        options = Object.assign({}, options, { fields: 'author,status,description,priority,tags,time,type,owner,execution' });

        const tasks = new Tasks([], options);
        const deferredTasks = tasks.update();

        app.stopListening(app, 'trigger:route:change');
        app.listenTo(app, 'trigger:route:change', value => {
            if (value !== 'route:home' && value !== 'router:task' && deferredTasks && deferredTasks.abort) {
                deferredTasks.abort();
            } else {
                app.t.forEach(item => {
                    item.xhr.abort();
                    item.promise.resolve(item.xhr);
                });
            }
        });

        return deferredTasks
            .then(() => {
                return tasks;
            });
    },

    singleTask(options) {
        return this.fetch(Task, options);
    },

    statuses() {
        return $.Deferred().resolve(Task.getPossibleStatuses()); // eslint-disable-line
    },

    statusUpdateIntervals() {
        return $.Deferred().resolve(Task.getStatusUpdateIntervals()); // eslint-disable-line
    },

    releaseStatuses() {
        return $.Deferred().resolve(Release.getPossibleStatuses()); // eslint-disable-line
    },

    platforms() {
        return this.clients().then(clients => {
            return clients.getPlatforms();
        });
    },

    tags() {
        return this.fetch(Tags);
    },

    dns() {
        return this.fetch(Dns);
    },

    architectures() {
        return $.Deferred().resolve(['linux','freebsd','osx']); // eslint-disable-line
    },

    priorities() {
        return $.Deferred().resolve([ // eslint-disable-line
            'BACKGROUND:LOW',
            'BACKGROUND:NORMAL',
            'BACKGROUND:HIGH',
            'SERVICE:LOW',
            'SERVICE:NORMAL',
            'SERVICE:HIGH',
            'USER:LOW',
            'USER:NORMAL',
            'USER:HIGH']);
    },

    models() {
        return this.clients().then(clients => {
            return clients.getCpuModels();
        });
    },

    hosts() {
        return this.clients().then(clients => {
            return clients.getHosts();
        });
    },

    types() {
        return this.fetch(TaskTypes);
    },

    resourceTypes() {
        return this.fetch(ResourceTypes);
    },

    singleResource(options) {
        return this.fetch(Resource, options);
    },

    resources(options) {
        const resources = new TaskResourcesCollection([], options);
        const deferredResources = resources.update();

        app.stopListening(app, 'trigger:route:change');
        app.listenTo(app, 'trigger:route:change', value => {
            if (value !== 'route:resources' && deferredResources && deferredResources.abort) {
                deferredResources.abort();
            }
        });

        return deferredResources
            .then(() => {
                return resources;
            });
    },

    /**
     * @param {Object}         options         - hash for users and task props and resources
     * @param {Backbone.Model} resourceLocator - application resource locator
     *
     * @returns {JQueryDeferred}
     */
    resourceList(options, resourceLocator) {
        const basicResources = $.Deferred(); // eslint-disable-line

        $.when(
            this.taskProps({}, resourceLocator),
            this.resources(options.resources),
            this.user(options.user || {}, resourceLocator)
        ).done((taskProps, resources, user) => {
            basicResources.resolve(resourceLocator ?
                resourceLocator.set({ resources }) :
                new CoreApplication({
                    resources,
                    props: taskProps,
                    user
                }));
        }).fail(function () {
            basicResources.rejectWith(this, arguments);
        });

        return basicResources;
    },

    schedulers(options) {
        const schedulers = new SchedulersCollection([], options);
        const deferredSchedulers = schedulers.update();

        app.stopListening(app, 'trigger:route:change');
        app.listenTo(app, 'trigger:route:change', value => {
            if (value !== 'route:schedulers' && deferredSchedulers && deferredSchedulers.abort) {
                deferredSchedulers.abort();
            } else {
                app.t.forEach(item => {
                    item.xhr.abort();
                    item.promise.resolve(item.xhr);
                });
            }
        });

        return deferredSchedulers
            .then(() => {
                return schedulers;
            });
    },

    schedulerStatuses() {
        return $.Deferred().resolve(Object.keys(SchedulerModel.STATUS)); // eslint-disable-line
    },

    schedulersProps(options, resourceLocator) {
        const props = $.Deferred(); // eslint-disable-line

        $.when(
            this.schedulerStatuses(),
            this.resourceTypes(),
            this.groups(),
            this.types()
        ).done((statuses, resourceTypes, groups, taskTypes) => {
            const propsToSet = {
                resourceTypes,
                statuses,
                groups,
                types: taskTypes
            };

            if (resourceLocator) {
                resourceLocator.setProps(propsToSet);
            }

            props.resolve(propsToSet);
        }).fail(function () {
            props.rejectWith(this, arguments);
        });

        return props;
    },

    schedulersList(options, resourceLocator) {
        const basicResources = $.Deferred(); // eslint-disable-line

        $.when(
            this.schedulersProps({}, resourceLocator),
            this.schedulers(options.schedulers),
            this.user(options.user || {}, resourceLocator)
        ).done((schedulerProps, schedulers, user) => {
            basicResources.resolve(resourceLocator ?
                resourceLocator.set({ schedulers }) :
                new CoreApplication({
                    schedulers,
                    props: schedulerProps,
                    user
                }));
        }).fail(function () {
            basicResources.rejectWith(this, arguments);
        });

        return basicResources;
    },

    singleScheduler(options) {
        return this.fetch(SchedulerModel, options);
    },

    scheduler(options, resourceLocator) {
        const singleSchedulerResources = $.Deferred(); // eslint-disable-line

        $.when(
            this.schedulersProps(options.props || {}, resourceLocator),
            this.singleScheduler(options.scheduler),
            this.user(options.user || {}, resourceLocator)
        )
            .done((props, scheduler, user) => {
                singleSchedulerResources.resolve(resourceLocator ?
                    resourceLocator.set({ scheduler }) :
                    new CoreApplication({
                        scheduler,
                        props,
                        user
                    }));
            })
            .fail(function () {
                singleSchedulerResources.rejectWith(this, arguments);
            });

        return singleSchedulerResources;
    },

    groups(options) {
        return this.fetch(UserGroups, [], options);
    },

    externalHelpers() {
        const host = window.develop ? '' : '//sandbox-ui.s3.mds.yandex.net';

        return $.getScript(host + '/externalHelpers.js');
    },

    releaseProps(options, resourceLocator) {
        const rprops = $.Deferred(); // eslint-disable-line

        $.when(
            this.resourceTypes(),
            this.releaseStatuses()
        ).done((types, statuses) => {
            const propsToSet = {
                resourceTypes: types,
                releaseStatuses: statuses
            };

            if (resourceLocator) {
                resourceLocator.setProps(propsToSet);
            }

            rprops.resolve(propsToSet);
        }).fail(function () {
            rprops.rejectWith(this, arguments);
        });

        return rprops;
    },

    taskProps(options, resourceLocator) {
        const self = this;
        const props = $.Deferred(); // eslint-disable-line
        const propsToLoad = this.prepareTaskPropsToLoad(resourceLocator);

        $.when.apply(window, propsToLoad)
            .done(function () {
                const propsToSet = {};
                const resolvedProps = arguments;

                self.taskPropsToLoad.forEach((propToLoad, index) => {
                    propsToSet[propToLoad.key] = resolvedProps[index];
                });

                if (resourceLocator) {
                    resourceLocator.setProps(propsToSet);
                }

                props.resolve(propsToSet);
            }).fail(function () {
                props.rejectWith(this, arguments);
            });

        return props;
    },

    prepareTaskPropsToLoad(resourceLocator) {
        const willBeLoaded = [];

        this.taskPropsToLoad.forEach(function (propToLoad) {
            if (resourceLocator.get(propToLoad.key)) {
                willBeLoaded.push($.Deferred().resolve(resourceLocator.get(propToLoad.key))); // eslint-disable-line
            } else {
                willBeLoaded.push(this[propToLoad.method]());
            }
        }, this);

        return willBeLoaded;
    },

    /**
     * @param {Object}         options         - hash for users and tasks
     * @param {Backbone.Model} resourceLocator - application resource locator
     *
     * @returns {JQueryDeferred}
     */
    home(options, resourceLocator) {
        const basicResources = $.Deferred(); // eslint-disable-line

        $.when(
            this.taskProps({}, resourceLocator),
            this.tasks(options.tasks),
            this.user(options.user || {}, resourceLocator)
        ).done((taskProps, tasks, user) => {
            basicResources.resolve(resourceLocator ? resourceLocator.set({ tasks }) :
                new CoreApplication({
                    user,
                    tasks,
                    props: taskProps
                }));
        }).fail(function () {
            basicResources.rejectWith(this, arguments);
        });

        return basicResources;
    },

    enqueuedTasks(options, resourceLocator) {
        const enqueudTaskResources = $.Deferred(); // eslint-disable-line

        $.when(
            this.user(options.user),
            this.enqueuedTasksList(options.enqueued)
        ).done((user, enqueuedTasks) => {
            enqueudTaskResources.resolve(resourceLocator ?
                resourceLocator.set({ enqueuedTasks }) :
                new CoreApplication({
                    user,
                    enqueuedTasks
                }));
        });

        return enqueudTaskResources;
    },

    enqueuedTasksList(options) {
        const enqueued = new EnqueuedTasksCollection([], options);

        return enqueued
            .update()
            .then(() => {
                return enqueued;
            });
    },

    taskTypesData(options, resourceLocator) {
        const basicResources = $.Deferred(); // eslint-disable-line

        $.when(
            this.taskProps(options.props || {}, resourceLocator),
            this.user(options.user || {}, resourceLocator)
        ).done((taskProps, user) => {
            basicResources.resolve(resourceLocator ? resourceLocator :
                new CoreApplication({
                    props: taskProps,
                    user
                }));
        }).fail(function () {
            basicResources.rejectWith(this, arguments);
        });

        return basicResources;
    },

    task(options, resourceLocator) {
        const singleTaskResources = $.Deferred(); // eslint-disable-line

        $.when(
            this.taskProps(options.props || {}, resourceLocator),
            this.singleTask(options.task),
            this.user(options.user || {}, resourceLocator)
        ).done((taskProps, task, user) => {
            singleTaskResources.resolve(resourceLocator ?
                resourceLocator.set({ task }) :
                new CoreApplication({
                    props: taskProps,
                    task,
                    user
                }));
        })
            .fail(function () {
                singleTaskResources.rejectWith(this, arguments);
            });

        return singleTaskResources;
    },

    releases(options) {
        const releases = new Releases([], options);
        const deferredReleases = releases.update();

        app.stopListening(app, 'trigger:route:change');
        app.listenTo(app, 'trigger:route:change', value => {
            if (value !== 'route:releases' && deferredReleases && deferredReleases.abort) {
                deferredReleases.abort();
            }
        });

        return deferredReleases
            .then(() => {
                return releases;
            });
    },

    releasesList(options, resourceLocator) {
        const basicResources = $.Deferred(); // eslint-disable-line

        $.when(
            this.taskProps({}, resourceLocator),
            this.releaseProps({}, resourceLocator),
            this.releases(options.releases),
            this.user(options.user || {}, resourceLocator)
        ).done((taskProps, releaseProps, releases, user) => {
            basicResources.resolve(resourceLocator ?
                resourceLocator.set({ releases }) :
                new CoreApplication({
                    releases,
                    props: _.extend({}, taskProps, releaseProps),
                    user
                }));
        }).fail(function () {
            basicResources.rejectWith(this, arguments);
        });

        return basicResources;
    },

    sandboxNotificationsList(options, resourceLocator) {
        const basicResources = $.Deferred(); // eslint-disable-line

        $.when(
            this.sandboxNotifications(options.notifications),
            this.taskProps(options.props || {}, resourceLocator),
            this.user(options.user || {}, resourceLocator)
        ).done((notifications, taskProps, user) => {
            basicResources.resolve(resourceLocator ?
                resourceLocator.set({ sandboxNotifications: notifications }) :
                new CoreApplication({
                    sandboxNotifications: notifications,
                    props: _.extend({}, taskProps),
                    user
                }));
        }).fail(function () {
            basicResources.rejectWith(this, arguments);
        });

        return basicResources;
    },

    sandboxNotifications(options) {
        const notifications = new SandboxNotificationsCollection([], options);

        return notifications
            .update()
            .then(() => {
                return notifications;
            });
    },

    serviceNotificationsList(options, resourceLocator) {
        const basicResources = $.Deferred(); // eslint-disable-line

        $.when(
            this.serviceNotifications(options.notifications || {}, resourceLocator),
            this.taskProps(options.props || {}, resourceLocator),
            this.user(options.user || {}, resourceLocator)
        ).done((notifications, taskProps, user) => {
            basicResources.resolve(resourceLocator ?
                resourceLocator.set({ serviceNotifications: notifications }) :
                new CoreApplication({
                    serviceNotifications: notifications,
                    props: _.extend({}, taskProps),
                    user
                }));
        }).fail(function () {
            basicResources.rejectWith(this, arguments);
        });

        return basicResources;
    },

    serviceNotifications() {
        return this.fetch(ServiceNotificationsCollection);
    },

    shards() {
        return this.fetch(ShardCollection);
    },

    shardsData(options, resourceLocator) {
        const shards = $.Deferred(); // eslint-disable-line

        $.when(
            this.user({}, resourceLocator),
            this.taskProps({}, resourceLocator),
            this.shards()
        )
            .done((user, props, shardsCollection) => {
                shards.resolve(resourceLocator ?
                    resourceLocator.set({ shards: shardsCollection }) :
                    new CoreApplication({
                        user,
                        props,
                        shards: shardsCollection
                    }));
            });

        return shards;
    },

    serviceThreads() {
        return this.fetch(ServiceThreadCollection);
    },

    serviceThreadsData(options, resourceLocator) {
        const srvThreads = $.Deferred(); // eslint-disable-line

        $.when(
            this.user({}, resourceLocator),
            this.taskProps({}, resourceLocator),
            this.serviceThreads()
        )
            .done((user, props, threads) => {
                srvThreads.resolve(resourceLocator ?
                    resourceLocator.set({ serviceThreads: threads }) :
                    new CoreApplication({
                        user,
                        props,
                        serviceThreads: threads
                    }));
            });

        return srvThreads;
    },

    resource(options, resourceLocator) {
        const singleResourceData = $.Deferred(); // eslint-disable-line

        $.when(
            this.taskProps(options.props || {}, resourceLocator),
            this.singleResource(options.resource),
            this.user(options.user || {}, resourceLocator)
        ).done((taskProps, resource, user) => {
            singleResourceData.resolve(resourceLocator ?
                resourceLocator.set({ resource }) :
                new CoreApplication({
                    props: taskProps,
                    resource,
                    user
                }));
        })
            .fail(function () {
                singleResourceData.rejectWith(this, arguments);
            });

        return singleResourceData;
    },

    resourceCreate(options, resourceLocator) {
        const newResourceData = $.Deferred(); // eslint-disable-line

        $.when(
            this.taskProps(options.taskProps, resourceLocator),
            this.user(options.user || {}, resourceLocator),
            this.resourceTypes()
        ).done((taskProps, user, resourceTypes) => {
            newResourceData.resolve(new CoreApplication({
                props: taskProps,
                user,
                types: resourceTypes
            }));
        })
            .fail(function () {
                newResourceData.rejectWith(this, arguments);
            });

        return newResourceData;
    },

    schedulerCreate(options, resourceLocator) {
        const singleSchedulerResources = $.Deferred(); // eslint-disable-line

        $.when(
            (options.id ? this.singleScheduler({ id: options.id }) : true),
            this.taskProps(options.taskProps, resourceLocator),
            this.user(options.user || {}, resourceLocator)
        ).done((scheduler, taskProps, user) => {
            const app = require('../app');

            if (scheduler !== true) {
                app.request('DRAFT_SCHEDULER', scheduler);
            }

            user.fetchGroups().done(() => {
                singleSchedulerResources.resolve(resourceLocator ?
                    resourceLocator.set({ scheduler: app.request('DRAFT_SCHEDULER') }) :
                    new CoreApplication({
                        scheduler: app.request('DRAFT_SCHEDULER'),
                        props: taskProps,
                        user
                    }));
            });
        }).fail(function () {
            singleSchedulerResources.rejectWith(this, arguments);
        });

        return singleSchedulerResources;
    },

    clientsMonitoring(options, resourceLocator) {
        const self = this;
        const clientsMonitoringResources = $.Deferred(); // eslint-disable-line

        $.when(this.tags()).done(tags => {
            $.when(
                self.taskProps({}, resourceLocator),
                self.clientsData({ options: Object.assign({}, options.clients, { _tags: tags }) }),
                self.user(options.user || {}, resourceLocator)
            ).done((taskProps, clients, user) => {
                clientsMonitoringResources.resolve(resourceLocator ?
                    resourceLocator.set({ clients }) :
                    new CoreApplication({
                        clients,
                        props: taskProps,
                        user,
                        _tags: tags
                    }));
            })
                .fail(function () {
                    clientsMonitoringResources.rejectWith(self, arguments);
                });
        });

        return clientsMonitoringResources;
    },

    /**
     * @param {Backbone.Model|Backbone.Collection} Class
     * @param [options]
     * @param [options2]
     * @returns {Object}
     */
    fetch(Class, options, options2) {
        const object = new Class(options, options2);

        return object.fetch({ reset: true })
            .then(() => {
                return object;
            });
    },

    /**
     * @param {Backbone.Model|Backbone.Collection} Class
     * @returns {JQueryDeferred}
     */
    fetchFailable(Class, options, options2) {
        const fetched = $.Deferred(); // eslint-disable-line
        const entity = new Class(options, options2);

        entity.fetch({ reset: true }).always(() => {
            fetched.resolve(entity);
        });

        return fetched;
    },

    clients() {
        return this.fetch(Clients);
    },

    clientsData(options) {
        const clients = new ClientsData([], options);
        const deferredClients = clients.update();

        app.stopListening(app, 'trigger:route:change');
        app.listenTo(app, 'trigger:route:change', value => {
            if (value !== 'route:clients' && deferredClients && deferredClients.abort) {
                deferredClients.abort();
            }
        });

        return deferredClients
            .then(() => {
                return clients;
            });
    },

    manageGroups(options, resourceLocator) {
        const mGroups = $.Deferred(); // eslint-disable-line

        $.when(
            this.user(options.user || {}, resourceLocator),
            this.taskProps({}, resourceLocator),
            this.groups({ isSuggest: false })
        )
            .done((user, props, groups) => {
                mGroups.resolve(resourceLocator ?
                    resourceLocator.set({ groups }) :
                    new CoreApplication({
                        user,
                        props,
                        groups
                    }));
            });

        return mGroups;
    },

    vaults() {
        return this.fetch(VaultsCollection);
    },

    semaphores(options) {
        const semaphores = new SemaphoresCollection([], options);

        return semaphores
            .update()
            .then(() => {
                return semaphores;
            });
    },

    manageVaults(options, resourceLocator) {
        const mVaults = $.Deferred(); // eslint-disable-line

        $.when(
            this.user(options.user || {}, resourceLocator),
            this.taskProps({}, resourceLocator),
            this.vaults()
        )
            .done((user, props, vaults) => {
                mVaults.resolve(resourceLocator ?
                    resourceLocator.set({ vaults }) :
                    new CoreApplication({
                        user,
                        props,
                        vaults
                    }));
            });

        return mVaults;
    },

    manageSemaphores(options, resourceLocator) {
        const mVaults = $.Deferred(); // eslint-disable-line

        $.when(
            this.user(options.user || {}, resourceLocator),
            this.taskProps({}, resourceLocator),
            this.semaphores(options.semaphores || {})
        )
            .done((user, props, semaphores) => {
                mVaults.resolve(resourceLocator ?
                    resourceLocator.set({ semaphores }) :
                    new CoreApplication({
                        user,
                        props,
                        semaphores
                    }));
            });

        return mVaults;
    },

    statistics(options, resourceLocator) {
        const stats = $.Deferred(); // eslint-disable-line

        $.when(
            this.user(options.user, resourceLocator),
            this.taskProps(options.props, resourceLocator)
        )
            .done((user, props) => {
                stats.resolve(resourceLocator ?
                    resourceLocator :
                    new CoreApplication({
                        user,
                        props
                    }));
            });

        return stats;
    }
};

[
    'user',
    'platforms',
    'tags',
    'statuses',
    'models',
    'hosts',
    'dns',
    'types',
    'resourceTypes',
    'clients',
    'externalHelpers'
].forEach(name => {
    resources[name] = _.memoize(resources[name]);
});

module.exports = resources;
