/* eslint max-statements: [1, 22] */

const app = require('./app');
const error = require('./error');
// eslint-disable-next-line no-unused-vars
const DateTimeIntervalModel = require('./model/DateTimeIntervalModel');

const Router = Backbone.Router.extend({

    routes: {
        '': 'toTaskList',
        'tasks(/)': 'home',
        'task/:id(/)': 'task',
        'task/:id/:action(/)': 'task',
        'clients(/)': 'clients',
        'releases(/)': 'releases',
        'resources(/)': 'resources',
        'resource/create(/)': 'resourceCreate',
        'resource/:id(/)': 'resource',
        'resource/:id/:action(/)': 'resource',
        'admin/:action(/:key)(/:sub)(/)': 'admin',
        'statistics(/:type)(/)': 'statistics',
        'schedulers(/)': 'schedulers',
        'scheduler/create(/)': 'schedulerCreate',
        'scheduler/:id/edit(/)': 'schedulerCreate',
        'scheduler/:id(/)': 'scheduler',
        'scheduler/:id/:action(/)': 'scheduler',
        'oauth(/)': 'oauth',
        'oauth/:action(/)': 'oauth',
        'error(/)': 'error',
        'notifications(/)': 'sandboxNotifications',
        'service-notifications(/)': 'serviceNotifications',
        'host/:id/queue(/)': 'hostQueue',
        'preferences(/)': 'preferences',
        '*default': 'notFound'
    },

    toTaskList() {
        this.navigate('tasks', {
            trigger: true
        });
    },

    /**
     * @param id Created task ID
     */
    toTaskCreate(id) {
        this.navigate('/task/' + id + '/view', { trigger: true });
    },

    toSchedulerCreate(id) {
        if (!id) {
            this.navigate('/schedulers', {
                trigger: true
            });
        }

        this.navigate('/scheduler/' + id + '/edit');
        this.trigger('route:schedulerCreate');
    },

    getResourceLocator() {
        const CoreApplication = require('./application/CoreApplication');
        const stored = app.reqres.request('RESOURCE_LOCATOR');

        if (stored) {
            return stored;
        }
        return app.reqres.request('RESOURCE_LOCATOR', new CoreApplication({}));
    },

    getLayoutConfig(configKey, propsToExtend) {
        if (!configKey) {
            throw new Error('Layout config key was not specified.');
        }

        const LayoutConfig = require('./application/config/layoutConfig');

        return LayoutConfig.getPageConfig(configKey, propsToExtend);
    },

    showBasicLayout(resourceLocator) {
        const BasicLayout = require('./views/layouts/BasicLayout');

        app.content.close();
        app.content.show(new BasicLayout({
            model: resourceLocator || this.getResourceLocator(),
            showContent: false
        }));
    },

    showFilterLayout() {
        if (!app.content.currentView) {
            this.showBasicLayout();
        }

        app.content.currentView.showFilter();
    },

    showContentLayout() {
        if (!app.content.currentView) {
            this.showBasicLayout();
        }

        app.content.currentView.showContent();
    },

    bindRouteRelatedEvents(routeName, appResources) {
        /* eslint complexity: [1, 8] */

        switch (routeName) {
            case 'home': {
                this.listenTo(appResources.getTasks(), 'sync filterChanged', this.onItemsCollectionSync);
                break;
            }
            case 'releases': {
                this.listenTo(appResources.getReleases(), 'sync filterChanged', this.onItemsCollectionSync);
                break;
            }
            case 'resources': {
                this.listenTo(appResources.getResources(), 'sync filterChanged', this.onItemsCollectionSync);
                break;
            }
            case 'schedulers': {
                this.listenTo(appResources.getSchedulers(), 'sync filterChanged', this.onItemsCollectionSync);
                break;
            }
            case 'task': {
                this.listenTo(appResources, 'change:pageProps', this.onTaskContentReplace);
                break;
            }
            case 'resource': {
                this.listenTo(appResources, 'change:pageProps', this.onResourceContentReplace);
                break;
            }
            case 'scheduler': {
                this.listenTo(appResources, 'change:pageProps', this.onSchedulerContentReplace);
                break;
            }
            default:
        }
    },

    unbindRouteRelatedEvents(routeName) {
        switch (routeName) {
            case 'task':
            case 'resource':
            case 'scheduler':
            default: {
                const resourceLocator = this.getResourceLocator();

                if (resourceLocator) {
                    this.stopListening(resourceLocator, 'change:pageProps');
                    this.stopListening(resourceLocator.getTasks(), 'sync');
                    this.stopListening(resourceLocator.getReleases(), 'sync');
                    this.stopListening(resourceLocator.getResources(), 'sync');
                    this.stopListening(resourceLocator.getSchedulers(), 'sync');
                }
            }
        }
    },

    /**
     * @param {TasksCollection|TaskResourcesCollection} itemsCollection
     */
    onItemsCollectionSync(itemsCollection) {
        /**
         * Can be generated in 2 cases:
         *  1. On real collection sync, in that case everything is OK.
         *  2. When just fetching a single task to set as draft and open in edit mode. In that case we don't need
         *     to update URL params.
         */
        if (itemsCollection instanceof Backbone.Collection) {
            const paramsToUpdate = _.extend(
                {},
                itemsCollection.getPagingOptions(),
                itemsCollection.getFilterParams().serializeAsUrl()
            );

            delete paramsToUpdate.total;

            // Arguments.length === 1 when user got on page with navigation link
            this.updateURL({ paramsToUpdate }, true, arguments.length === 1);
        }
    },

    onTaskContentReplace(appResources) {
        const props = appResources.getPageProps();

        if (props) {
            if (props.type === 'SINGLE_TASK' || props.type === 'SINGLE_TASK_CREATE') {
                this.updateURL({
                    pathFragment: (
                        '/task/' +
                        appResources.getTask().get('id') +
                        '/' +
                        appResources.getPageProps().action
                    )
                });
            }
        }
    },

    onResourceContentReplace(appResources) {
        const props = appResources.getPageProps();

        if (props) {
            if (props.type === 'SINGLE_RESOURCE' || props.type === 'CREATE_RESOURCE') {
                this.updateURL({
                    pathFragment: (
                        '/resource/' +
                        appResources.getResource().get('id') +
                        '/' +
                        appResources.getPageProps().action
                    )
                });
            }
        }
    },

    onSchedulerContentReplace(appResources) {
        const props = appResources.getPageProps();

        if (props) {
            if (props.type === 'SINGLE_SCHEDULER') {
                this.updateURL({
                    pathFragment: (
                        '/scheduler/' +
                        appResources.getScheduler().get('id') + '/' +
                        appResources.getPageProps().action
                    )
                });
            }
        }
    },

    buildQueryJson(params) {
        const keys = Object.keys(params);
        const result = {};

        keys.forEach(key => {
            const value = params[key];

            if ((value !== '' && value !== null) || (value instanceof Array && value.length)) {
                result[key] = value;
            }
        });

        return result;
    },

    buildQueryString(params) {
        const queryItems = _.map(params, (value, key) => {
            if (key && value) {
                return key + '=' + encodeURIComponent(value);
            }

            return '';
        });

        return _.filter(queryItems, item => {
            return item !== '';
        }).join('&');
    },

    parseQueryString(query) {
        const params = {};

        if (query[0] === '?') {
            query = query.substr(1);
        }

        query.split('&').forEach(part => {
            if (part !== '') {
                const match = part.split('=');

                params[match[0]] = decodeURIComponent(_.rest(match).join('='));
            }
        });

        return params;
    },

    getQueryParam(key) {
        const query = this.parseQueryString(location.search);

        if (_.isUndefined(key)) {
            return query;
        }
        return query[key];
    },

    /**
     * @description Navigates to the current path but with updated param set.
     *
     * @param {Object}  options
     * @param {Boolean} hardUpdate     If set to true all current query string in URL will be replaced with
     * @param {Boolean} replace
     */
    updateURL(options, hardUpdate, replace) {
        let params = {};
        let currentParams = {};
        let targetPath = '';
        let query = '';

        hardUpdate = (typeof hardUpdate === 'boolean' ? hardUpdate : false);
        replace = replace || false;

        if (!options.paramsToUpdate) {
            options.paramsToUpdate = {};
        }

        currentParams = this.parseQueryString(window.location.search);

        Object.keys(options.paramsToUpdate).forEach(key => {
            currentParams[key] = options.paramsToUpdate[key];
        });

        params = currentParams;

        if (hardUpdate === true) {
            _.difference(_.keys(params), _.keys(options.paramsToUpdate)).forEach(key => {
                delete params[key];
            });
        }

        query = this.buildQueryString(params);

        targetPath += (options.pathFragment ? options.pathFragment : this.getCurrentPathFragment());
        targetPath += (query === '' ? '' : ('?' + query));

        this.navigate(targetPath, { trigger: false, replace });
    },

    /**
     * @returns {String}
     */
    getCurrentPathFragment() {
        return window.location.pathname;
    },

    showErrorPage(xhr, name, description) {
        app.reqres.request(
            'ERR_TRACE',
            description && description.stack ?
                description.stack :
                xhr
        );
        error.showErrPage({
            xhr,
            name,
            description
        });
    }
});

module.exports = new Router();

// All controllers
require('./controller/home');
require('./controller/task');
require('./controller/error');
require('./controller/admin');
require('./controller/oauth');
require('./controller/notFound');
require('./controller/releases');
require('./controller/resource');
require('./controller/resourceCreate');
require('./controller/resources');
require('./controller/scheduler');
require('./controller/hostQueue');
require('./controller/schedulers');
require('./controller/statistics');
require('./controller/schedulerCreate');
require('./controller/clientsMonitoring');
require('./controller/serviceNotifications');
require('./controller/sandboxNotifications');
require('./controller/preferences');
