const error = require('../../../error');
const TaskTypeModel = require('../../../model/resources/TaskTypeModel');
const RestHostItemView = require('./RestHostItemView');
const UsedHostItemView = require('./UsedHostItemView');
const BasicConfirmDialogView = require('../../components/BasicConfirmDialogView');

/**
 * @class TaskTypeEditModalView
 * @extends BasicConfirmDialogView
 *
 * @property {TaskTypeModel} model
 * @property {Marionette.Region} restHosts
 * @property {Marionette.Region} usedHosts
 */
const TaskTypeEditModalView = BasicConfirmDialogView.extend({

    template: require('./tpl/TaskTypeEditModalView.hbs'),

    ui: {
        restHostsLengthCnt: '.rest_hosts_length',
        usedHostsLengthCnt: '.used_hosts_length',
        restHostsItems: '.task-cols__i_add',
        usedHostsItems: '.task-cols__i_remove'
    },

    events: _.extend({}, BasicConfirmDialogView.prototype.events, {
        'click @ui.restHostsItems': 'moveRestToUsed',
        'click @ui.usedHostsItems': 'moveUsedToRest'
    }),

    toAdd: [],
    toRemove: [],

    initialize() {
        BasicConfirmDialogView.prototype.initialize.apply(this, arguments);

        const usedHosts = this.model.getHosts(TaskTypeModel.HOSTS_TYPE.BOUND);

        this.restHostsCollection = new Backbone.Collection(_.map(_.difference(this.options.allHosts, usedHosts), host => {
            return { hostname: host };
        }), { comparator: 'hostname' });
        this.usedHostsCollection = new Backbone.Collection(_.map(usedHosts, host => {
            return { hostname: host };
        }), { comparator: 'hostname' });

        this.listenTo(this.restHostsCollection, 'remove', this.moveRestToUsed);
        this.listenTo(this.usedHostsCollection, 'remove', this.moveUsedToRest);

        this.listenTo(this.restHostsCollection, 'add', this.updateCounters);
        this.listenTo(this.usedHostsCollection, 'add', this.updateCounters);
    },

    onRender() {
        this.restHosts.close();
        this.usedHosts.close();
        this.restHosts.show(this.getRestHostsView());
        this.usedHosts.show(this.getUsedHostsView());

        this.updateCounters();
    },

    getRestHostsView() {
        return new Marionette.CollectionView({
            collection: this.restHostsCollection,
            className: 'task-cols__left',
            itemView: RestHostItemView,
            itemViewOptions: {
                _collection: this.restHostsCollection
            }
        });
    },

    getUsedHostsView() {
        return new Marionette.CollectionView({
            collection: this.usedHostsCollection,
            className: 'task-cols__right',
            itemView: UsedHostItemView,
            itemViewOptions: {
                _collection: this.usedHostsCollection
            }
        });
    },

    onApprove() {
        const self = this;
        const usedHosts = this.usedHostsCollection.map(host => {
            return host.get('hostname');
        });

        $.ajax({
            url: ('/api/v1.0/service/binding/' + this.model.get('type')),
            type: 'PUT',
            data: JSON.stringify(usedHosts),
            dataType: 'json',
            contentType: 'application/json'
        })
            .fail(() => {
                error.message('There was a problem during binding/unbinding.', {
                    autoClose: true,
                    reload: false,
                    close: false
                });
            })
            .always(() => {
                self.close();
            });
    },

    moveRestToUsed(model) {
        this.moveItem(model, this.usedHostsCollection, this.toRemove, this.toAdd);
    },

    moveUsedToRest(model) {
        this.moveItem(model, this.restHostsCollection, this.toAdd, this.toRemove);
    },

    moveItem(modelToMove, toCollection, tempArrayFrom, tempArrayTo) {
        const hostname = modelToMove.get('hostname');
        const addIndex = tempArrayFrom.indexOf(hostname);

        if (addIndex !== -1) {
            tempArrayFrom.splice(addIndex, 1);
        }

        tempArrayTo.push(hostname);
        toCollection.add(modelToMove);

        this.updateCounters();
    },

    updateCounters() {
        this.ui.restHostsLengthCnt.text(this.restHostsCollection.length);
        this.ui.usedHostsLengthCnt.text(this.usedHostsCollection.length);
    },

    onClose() {
        this.stopListening(this.restHostsCollection);
        this.stopListening(this.usedHostsCollection);
    }
});

module.exports = TaskTypeEditModalView;
