/* eslint camelcase: 0 */

// eslint-disable-next-line no-unused-vars
const error = require('../../../error');
const OwnerSuggestView = require('../../controlSection/filter/suggest/OwnerSuggestView');
const BasicConfirmDialogView = require('../../components/BasicConfirmDialogView');
const NotificationRecipientSuggestView = require('../../createTask/NotificationRecipientSuggestView');

/**
 * @class VaultEditModalView
 * @extends BasicConfirmDialogView
 *
 * @property {VaultModel}        model
 * @property {Marionette.Region} shared
 * @property {Marionette.Region} ownerSelector - Will exist only if current user is admin. Admins can create
 *                                               schedulers with any owner, regular users can set as owner only
 *                                               themselfs and groups in which them are members.
 */
const VaultEditModalView = BasicConfirmDialogView.extend({

    template: require('./tpl/VaultEditModalView.hbs'),

    ui: {
        owner: '.vault_owner select',
        name: '.vault_name',
        data: '.vault_data',
        description: '.vault_description'
    },

    events: {
        'change @ui.owner': 'onOwnerChange',
        'bemmy:refresh @ui.name': 'onNameChange',
        'bemmy:refresh @ui.data': 'onDataChange',
        'bemmy:refresh @ui.description': 'onDescriptionChange',
        approve: 'onApprove'
    },

    initialize() {
        BasicConfirmDialogView.prototype.initialize.apply(this, arguments);

        this._model = this.model.clone();

        this.shareUnits = new Backbone.Model({
            recipients: this._model
                .get('shared')
                .map(sharedInit => {
                    return sharedInit.get('name');
                })
        });

        this.onNameChange = _.debounce(this.onNameChange, 0);
        this.onDataChange = _.debounce(this.onDataChange, 0);
        this.onDescriptionChange = _.debounce(this.onDescriptionChange, 0);
    },

    onRender() {
        this.$('.input').bemmyInput();
        this.$('.select').bemmySelect();
        this.$('.textarea').bemmyTextarea();

        this.shared.close();
        this.shared.show(new NotificationRecipientSuggestView({
            ownerGroups: this.options.taskProps.groups.convertToList(),
            model: this.shareUnits
        }));

        if (this.options.user.isAdmin()) {
            this.ownerSelector.close();
            this.ownerSelector.show(new OwnerSuggestView({
                model: this._model,
                ownerGroups: this.options.taskProps.groups.convertToList(),
                defaultValue: this._model.get('owner')
            }));
        }
    },

    onApprove() {
        const previousAttributes = this.model.toJSON();

        this.onSharedUnitsChange();

        return this.model
            .set(this._model.attributes)
            .save()
            .then(() => {
                this.close();
            })
            .fail(err => {
                error.fromXHR(err);
                this.model.set(previousAttributes);
            });
    },

    onNameChange() {
        const value = this.ui.name.find('input').val() || '';

        this._model.set('name', value.trim());
    },

    onDescriptionChange() {
        this._model.set('description', this.ui.description.find('textarea').val());
    },

    onOwnerChange() {
        this._model.set('owner', this.ui.owner.val());
    },

    onDataChange() {
        const data = this.ui.data.find('textarea').val();

        this._model.set({
            data,
            data_length: data.length
        });
    },

    onSharedUnitsChange() {
        this._model.get('shared').set(_.map(this.shareUnits.get('recipients'), sharedUnit => {
            return new Backbone.Model({ name: sharedUnit });
        }));

        this._model.trigger('change');
    },

    serializeData() {
        const serialized = this._model.toJSON();

        serialized.dataPlaceholder = ('Hidden data [' + serialized.data_length + ' bytes length]');
        serialized.user = {
            login: this.options.user.get('login'),
            groups: this.options.user.get('groups').map(group => {
                return group.get('name');
            }),
            isAdmin: this.options.user.isAdmin()
        };

        return serialized;
    }
});

module.exports = VaultEditModalView;
