const TaskRemoveConfirmView = require('../../tasks/taskRemoveConfirm/TaskRemoveConfirmView');
const BasicControllerBehavior = require('./BasicControllerBehavior');
const router = require('../../../router');
const error = require('../../../error');

/**
 * @class TaskControllerBehavior
 * @extends BasicControllerBehavior
 */
const TaskControllerBehavior = BasicControllerBehavior.extend({
    ui: _.extend({}, BasicControllerBehavior.prototype.ui, {
        execute: '.execute_trigger',
        stop: '.stop_trigger',
        priorInc: '.prior_trigger',
        suspend: '.suspend_trigger',
        duplicate: '.duplicate_trigger'
    }),

    events: _.extend({}, BasicControllerBehavior.prototype.events, {
        'click @ui.execute': 'onExecute',
        'click @ui.stop': 'onStop',
        'click @ui.priorInc': 'onPriorInc',
        'click @ui.suspend': 'onSuspend',
        'click @ui.duplicate': 'onDuplicate'
    }),

    defaults: {
        entityName: 'task',
        draftItemSetterName: 'setDraftTask',
        removeConfirmDialogClass: TaskRemoveConfirmView
    },

    initialize(options) {
        this.options = _.extend({}, this.options, options, {
            loadbutton: 'button_load'
        });

        this.listenTo(Backbone, 'task:exec', this.onTaskExec);
        this.listenTo(Backbone, 'task:exec:finished', this.onTaskExecFinished);
        this.listenTo(Backbone, 'task:stop', this.onTaskStop);
        this.listenTo(Backbone, 'task:stop:finished', this.onTaskStopFinished);
        this.listenTo(Backbone, 'task:suspend', this.onTaskSuspend);
        this.listenTo(Backbone, 'task:suspend:finished', this.onTaskSuspendFinished);
        this.listenTo(Backbone, 'task:duplicate', this.onTaskDuplicate);
        this.listenTo(Backbone, 'task:duplicate:finished', this.onTaskDuplicateFinished);
        this.listenTo(Backbone, 'task:increasePriority', this.onTaskIncreasePriority);
        this.listenTo(Backbone, 'task:increasePriority:finished', this.onTaskIncreasePriorityFinished);
    },

    onTaskExec() {
        this.ui.execute.addClass(this.options.loadbutton);
    },

    onTaskExecFinished() {
        this.ui.execute.removeClass(this.options.loadbutton);
    },

    onTaskStop() {
        this.ui.stop.addClass(this.options.loadbutton);
    },

    onTaskStopFinished() {
        this.ui.stop.removeClass(this.options.loadbutton);
    },

    onTaskSuspend() {
        this.ui.suspend.addClass(this.options.loadbutton);
    },

    onTaskSuspendFinished() {
        this.ui.suspend.removeClass(this.options.loadbutton);
    },

    onTaskDuplicate() {
        this.ui.duplicate.addClass(this.options.loadbutton);
    },

    onTaskDuplicateFinished() {
        this.ui.duplicate.removeClass(this.options.loadbutton);
    },

    onTaskIncreasePriority() {
        this.ui.priorInc.addClass(this.options.loadbutton);
    },

    onTaskIncreasePriorityFinished() {
        this.ui.priorInc.removeClass(this.options.loadbutton);
    },

    onExecute(evtObj) {
        evtObj.preventDefault();

        Backbone.trigger('task:exec');

        if (!this.view.model.editMode) {
            this.view.model.taskAction().done(() => {
                Backbone.trigger('task:exec:finished');
            });
        }
    },

    onStop(evtObj) {
        evtObj.preventDefault();

        Backbone.trigger('task:stop');

        if (!this.view.model.editMode) {
            this.view.model.taskStopAction().done(() => {
                Backbone.trigger('task:stop:finished');
            });
        }
    },

    onSuspend(evtObj) {
        evtObj.preventDefault();

        Backbone.trigger('task:suspend');

        if (!this.view.model.editMode) {
            this.view.model.taskSuspend().done(() => {
                Backbone.trigger('task:suspend:finished');
            });
        }
    },

    onPriorInc(evtObj) {
        evtObj.preventDefault();

        Backbone.trigger('task:increasePriority');

        if (!this.view.model.editMode) {
            const result = this.view.model.increasePriority();

            if (result) {
                result.done(() => {
                    Backbone.trigger('task:increasePriority:finished');
                });
            } else {
                Backbone.trigger('task:increasePriority:finished');
            }
        }

        return false;
    },

    onDuplicate(evtObj) {
        evtObj.preventDefault();

        Backbone.trigger('task:duplicate');

        if (!this.view.model.duplicate) {
            throw new Error('Controlled entity does not implement .duplicate() method');
        }

        let windowReference;
        const isMetaKey = evtObj.ctrlKey || evtObj.metaKey || evtObj.which === 2;

        if (isMetaKey) {
            windowReference = window.open();
        }

        this.view.model
            .duplicate()
            .done(response => {
                Backbone.trigger('task:duplicate:finished');

                const url = '/task/' + response.id + '/view';

                if (isMetaKey && windowReference) {
                    windowReference.location = url;
                } else {
                    router.navigate('/task/' + response.id + '/view', { trigger: true });
                }
            })
            .fail(() => {
                Backbone.trigger('task:duplicate:finished');

                if (isMetaKey) {
                    windowReference.close();
                }
                error.message('Error during task duplication.');
            });
    }
});

module.exports = TaskControllerBehavior;
