/**
 * Utilize keyboard navigation and menu control.
 *
 * @class SuggestNavigationBehavior
 * @extends Marionette.Behavior
 */
const SuggestNavigationBehavior = Marionette.Behavior.extend({

    defaults: {
        itemActiveClass: 'sel',
        valueAttrName: 'suggest-value'
    },

    ui: {
        suggestItems: $(),
        input: '.tag__input'
    },

    events: {
        keydown: 'onKeyNavigate',
        click: 'onItemSelect'
    },

    collectionEvents: {
        'change:items': 'updateUIElements'
    },

    onKeyNavigate(evtObj) {
        /* eslint max-statements: [1, 13] */
        /* eslint complexity: [1, 9] */

        if ([CONST.KEYS.ENTER, CONST.KEYS.UP, CONST.KEYS.DOWN, CONST.KEYS.TAB].includes(evtObj.keyCode)) {
            evtObj.preventDefault();

            const selected = this.ui.suggestItems.filter('.' + this.options.itemActiveClass);

            switch (evtObj.keyCode) {
                case CONST.KEYS.ENTER:
                case CONST.KEYS.TAB: {
                    const suggestItems = this.ui.suggestItems;

                    if (selected.length) {
                        this.setSelected(selected);
                    } else if (suggestItems.length && this.ui.input.val()) {
                        this.setSelected(suggestItems.eq(0));
                    }
                    break;
                }
                case CONST.KEYS.UP: {
                    this.setActiveItem(selected, 'prev');
                    break;
                }
                case CONST.KEYS.DOWN: {
                    this.setActiveItem(selected, 'next');
                    break;
                }
                default:
            }
        }
    },

    onItemSelect(evtObj) {
        evtObj.preventDefault();

        const selectedItem = $(evtObj.target).closest('li');

        if (selectedItem.length > 0) {
            this.setSelected(selectedItem);
        }
    },

    setActiveItem(currentItem, direction) {
        let targetIndex = 0;
        let targetItem = null;
        const index = this.ui.suggestItems.index(currentItem);

        direction = ((direction === 'next' || direction === 'prev') ? direction : 'next');

        if (currentItem.length > 0) {
            targetIndex = (index + (direction === 'next' ? 1 : -1));
            targetIndex = ((targetIndex >= this.ui.suggestItems.length || targetIndex < 0) ? 0 : targetIndex);

            targetItem = this.ui.suggestItems.eq(targetIndex);
        } else {
            targetItem = this.ui.suggestItems.eq(targetIndex);
        }

        if (targetItem.length > 0) {
            currentItem.removeClass(this.options.itemActiveClass);
            targetItem.addClass(this.options.itemActiveClass);
        }

        return targetItem;
    },

    setSelected(currentItem) {
        const selectedValue = currentItem ? this.getSelectedItemValue(currentItem) : '';
        const recipients = this.view.model.attributes.recipients;

        if (recipients && recipients.includes(selectedValue)) {
            this.ui.input.val('');
        } else {
            this.view.trigger('selected', {
                value: selectedValue,
                type: currentItem.get(0).dataset.suggestType
            });

            setTimeout(() => {
                this.view.model.attributes.recipients = _.uniq(this.view.model.attributes.recipients);
            }, 250);
        }
    },

    getSelectedItemValue(currentItem) {
        let value = currentItem.data(this.options.valueAttrName);

        if (typeof value === 'undefined') {
            value = $.trim(currentItem.text());
        }

        return value;
    },

    updateUIElements() {
        this.ui.suggestItems = this.$('.autocomplete_item');
    }
});

module.exports = SuggestNavigationBehavior;
