const router = require('../../router');
const FormAcceptBehavior = require('../../views/behaviors/forms/FormAcceptBehavior');
const ClientFilterModel = require('../../model/clients/ClientFilterModel');

/**
 * @class ClientsFilterLayout
 * @extends Backbone.Marionette.Layout
 */
const ClientsFilterLayout = Marionette.Layout.extend({

    className: 'section__ii section__ii_filter',
    template: require('./tpl/ClientsFilterLayout.hbs'),

    ui: {
        busy: '.is_busy select',
        apply: '.apply',
        reset: '.reset',
        alive: '.is_alive select',
        space: '.is_enough_space select',
        platform: '.platform select',
        searchPhrase: '.search_phrase',
        searchPhraseInput: '.search_phrase input',
        searchTags: '.search_tags',
        searchTagsInput: '.search_tags input'
    },

    events: {
        'click @ui.apply': 'onApply',
        'click @ui.reset': 'onReset',
        'change @ui.busy': 'onBusyChange',
        'change @ui.alive': 'onAliveChange',
        'change @ui.space': 'onSpaceChange',
        'change @ui.platform': 'onPlatformChange',
        'bemmy:refresh @ui.searchPhrase': 'onSearchPhraseChange',
        'bemmy:refresh @ui.searchTags': 'onTagsChange'
    },

    collectionEvents: {
        filter: 'onCollectionFilterUpdate'
    },

    behaviors: {
        FormAcceptBehavior: {}
    },

    initialize(options) {
        this.options = _.extend({}, this.options, options);

        this.onSearchPhraseChange = _.debounce(this.onSearchPhraseChange.bind(this), 250);
        this.onTagsChange = _.debounce(this.onTagsChange.bind(this), 250);
        this.listenTo(this, FormAcceptBehavior.EVENTS.ACCEPT_EVENT, this.onApply);
    },

    onRender() {
        const self = this;

        this.$('.input').bemmyInput();
        this.$('.select').bemmySelect();
        this.$('.button').bemmyButton();

        setTimeout(() => {
            self.markApplyBtn(false);
        }, 250); // Debounce time
    },

    onApply() {
        const self = this;
        const query = router.buildQueryString(this.model.serializeAsQuery());

        if (query) {
            router.navigate('/clients?' + query, { trigger: false });
        } else {
            router.navigate('/clients', { trigger: false });
        }

        this.collection.updateFilter(client => {
            /* eslint max-statements: [1, 23] */
            /* eslint complexity: [1, 9] */

            const result = true;

            if (!client) {
                return -1;
            }

            if (self.model.get('normalFreeSpace') !== '' &&
                self.model.get('normalFreeSpace') !== client.get('disk').status) {
                return false;
            }

            const searchPhrase = self.model.get('searchPhrase');
            let reg = null;

            try {
                reg = new RegExp(searchPhrase, 'igm');
            } catch (excpt) {
                /* eslint no-console: 0 */
                console.log('reg exp compilation failed');

                return false;
            }

            if (reg && !reg.test(_.values(client.attributes).join('|'))) {
                return false;
            }

            return result;
        });
    },

    onCollectionFilterUpdate(collection, filter) {
        if (filter && _.isFunction(filter) && filter() !== -1) {
            this.onReset();
        }

        const filterValues = this.model.clone();

        this.collection.collection.setFilterParams(filterValues);

        this.markApplyBtn(false);
    },

    onBusyChange(evtObj) {
        this.model.set('busy', this.getBinaryValue(evtObj.target.value));
        this.markApplyBtn();
    },

    onAliveChange(evtObj) {
        this.model.set('alive', this.getBinaryValue(evtObj.target.value));
        this.markApplyBtn();
    },

    onSpaceChange() {
        this.model.set('normalFreeSpace', this.ui.space.val());
        this.markApplyBtn();
    },

    onPlatformChange(evtObj) {
        this.model.set('platform', evtObj.target.value);
        this.markApplyBtn();
    },

    onSearchPhraseChange() {
        this.model.set('searchPhrase', this.ui.searchPhraseInput.val());
        this.markApplyBtn();
    },

    onTagsChange() {
        this.model.set('tags', this.ui.searchTagsInput.val());
        this.markApplyBtn();
    },

    onReset() {
        this.model = new ClientFilterModel();
        this.render();
        this.onApply();
    },

    getBinaryValue(value) {
        return (value === 'true' ? true : (value === 'false' ? false : null));
    },

    serializeData() {
        return _.extend({}, this.model.toJSON(), this.options.props);
    },

    markApplyBtn(toggle) {
        toggle = _.isUndefined(toggle);

        this.ui.apply.toggleClass('button_type_hl', toggle);
    },

    onClose() {
        this.stopListening(this, FormAcceptBehavior.EVENTS.ACCEPT_EVENT);
    }
});

module.exports = ClientsFilterLayout;
