/**
 * @class ClientsGroupView
 * @extends Backbone.Marionette.ItemView
 *
 * @property {ClientsGroupModel} model
 * @property {} collection
 */
const ClientsGroupView = Marionette.ItemView.extend({

    template: require('./tpl/ClientsGroupView.hbs'),

    ui: {
        check: '.check',
        commentInput: '.input__input',
        submitComment: '.input__submit',
        commentEditModeSwitcher: '.button_clients-comment'
    },

    events: {
        'click @ui.commentEditModeSwitcher': 'onEdit',
        'click @ui.submitComment': 'onSubmit',
        'keydown @ui.commentInput': 'onKeyDown',
        'bemmy:set @ui.check': 'onSelect'
    },

    collectionEvents: {
        'change:selected': 'onChange',
        'change:msg': 'onMsgChange'
    },

    initialize(options) {
        this.options = _.extend({}, this.options, options);
    },

    onRender() {
        this.ui.check.bemmyCheck();
        this.$('.button').bemmyButton(); // Monitoring links popup
    },

    onShow() {
        if (this.options.mode === 'compact') {
            const self = this;

            setTimeout(() => {
                self.$('.clients__i').each((index, elt) => {
                    if ((document.body.clientWidth - elt.offsetLeft) < 350) {
                        $(elt).find('.clients__pop').css({ right: '-1px', left: 'auto' });
                    }
                });
            }, 0);
        }
    },

    onSelect(evtObj, value) {
        const client = this.getClient(this.getClientCnt(evtObj));

        if (client) {
            client.set({ selected: value }, { key: 'selected', value });
        }
    },

    onEdit(evtObj) {
        this.getClientCnt(evtObj).find('.clients__comment').toggleClass('clients__comment_edit');
    },

    onSubmit(evtObj) {
        const container = this.getClientCnt(evtObj);
        const client = this.getClient(container);
        const value = container.find('.input__input').val();

        if (client) {
            client.set('msg', value);
        }

        container.find('.clients__comment__text__i').text(value ? value : 'NO comment');
        container.find('.clients__comment').toggleClass('clients__comment_edit');
    },

    onKeyDown(evtObj) {
        if (evtObj.keyCode === 13) {
            this.onSubmit(evtObj);
        }
    },

    onChange(model, selected, opts) {
        if (opts.source === 'controls') {
            const bemmyCheck = this.$('#' + model.get('id')).find('.check').data('bemmy');

            if (bemmyCheck) {
                bemmyCheck.setChecked(model.get('selected'));
            }
        }
    },

    onMsgChange(model) {
        const container = this.getClientCntBasedOnModel(model);
        const msg = this.getClientMsg(model);

        container.find('.clients__comment__text__i').text(msg);
        container.find('.input__input').val(msg);
    },

    getClientMsg(client) {
        const msg = client.get('msg');

        return (msg ? msg : 'NO comment');
    },

    getClientCntBasedOnModel(client) {
        return this.$('#' + client.get('id'));
    },

    getClientCnt(evtObj) {
        return $(evtObj.target).closest('.clients__i').eq(0);
    },

    getClient(container) {
        return this.collection.findWhere({ id: container.data('id') });
    },

    serializeData() {
        const subs = this.model.get('subs');

        if (subs) {
            const keys = Object.keys(subs);

            keys.forEach(key => {
                subs[key] = subs[key].map(sub => {
                    return sub.serializeData ? sub.serializeData() : sub;
                });
            });
        }

        return _.extend(
            {
                isCompact: this.options.mode === 'compact',
                isAdmin: this.options.user.isAdmin()
            },
            this.model.toJSON(),
            { clients: this.collection.map(clientItem => {
                return clientItem.serializeData();
            }) },
            subs ? { subs } : {}
        );
    }
});

module.exports = ClientsGroupView;
