const SuggestGroupsCollection = require('../../collection/suggest/SuggestGroupsCollection');
const SuggestGroupView = require('../controlSection/filter/suggest/SuggestGroupView');
const layoutMapper = require('../mixin/layoutMapper');

/**
 * @class BasicSuggestView
 * @extends Marionette.CompositeView
 * @mixes layoutMapper
 *
 * @property {Backbone.Model} model - In most of cases will be the model of some filter view
 */
const BasicSuggestView = Marionette.CompositeView.mixin(layoutMapper).extend({

    template: require('./tpl/BasicSuggestView.hbs'),

    itemView: SuggestGroupView,
    itemViewContainer: '.menu',

    behaviors: {
        SuggestNavigationBehavior: {},
        ClickOut: { eventName: 'clickOut' }
    },

    ui: {
        inputCnt: '.input',
        input: '.input__input'
    },

    events: {
        'input @ui.input': 'debouncedOnInput',
        'bemmy:refresh @ui.inputCnt': 'debouncedOnRefresh'
    },

    collectionEvents: {
        'change:items': 'onCollectionFetched'
    },

    options: {
        opened: false,
        openedClass: 'select_open',
        placeholder: '',
        defaultValue: '',
        clearOnSelect: false,

        modelTargetKey: '__basic_suggest_value__'
    },

    initialize(options) {
        this.options = _.extend({}, this.options, options);
        this.collection = new SuggestGroupsCollection();

        this.debouncedOnInput = this.onInput.bind(this);
        this.debouncedOnRefresh = this.onRefresh.bind(this);

        const self = this;

        setTimeout(() => {
            self.setSuggestGroups();
        }, 50);

        this.on('selected', this.onItemSelect);

        if (this.model && this.options.modelTargetKey) {
            this.listenTo(this.model, 'change:' + this.options.modelTargetKey, this.mapModelToView);
        }
    },

    onRender() {
        this.ui.inputCnt.bemmyInput();
    },

    toggleMenu(toggle) {
        const show = (typeof toggle === 'undefined' ? !this.options.opened : toggle);

        if (show) {
            this.showMenu();
        } else {
            this.hideMenu();
        }
    },

    showMenu() {
        this.options.opened = true;
        this.$el.addClass(this.options.openedClass);
    },

    hideMenu() {
        this.options.opened = false;
        this.$el.removeClass(this.options.openedClass);
    },

    onCollectionFetched() {
        this.collection.trigger('reset');

        if (!this.options.opened) {
            this.toggleMenu();
        }
    },

    setSuggestGroups() {
        throw new Error('SANDBOX: Each suggest should set groups.');
    },

    serializeData() {
        return {
            value: this.options.defaultValue,
            placeholder: this.options.placeholder
        };
    },

    setValue(valueToSet) {
        this.ui.input.val(valueToSet);
        this.ui.input.change();
        this.toggleMenu(Boolean(valueToSet));
    },

    setModelValue(valueToSet, options) {
        if (this.model) {
            this.model.set(this.options.modelTargetKey, valueToSet, options);
        }
    },

    onRefresh() {
        if (this.ui.input.val) {
            this.setModelValue(this.ui.input.val());
        }
    },

    onInput(evtObj) {
        let value = evtObj.target.value;

        if (value.length > 0 && value.charCodeAt(0) > 255) {
            value = this._convertToEngLayout(value.toLowerCase());
        }

        this.collection.fetchSuggestGroups({
            query: value
        });
    },

    onItemSelect(selectedData) {
        const valueToSet = (this.options.clearOnSelect ? '' : selectedData.value);

        this.setModelValue(valueToSet);
        this.hideMenu();
    },

    onClickOut() {
        this.hideMenu();
    },

    mapModelToView() {
        if (this.model) {
            this.setValue(this.model.get(this.options.modelTargetKey));
        }
    },

    onClose() {
        this.stopListening(this, 'selected');
        this.stopListening(this.model);
    }
});

module.exports = BasicSuggestView;
