import BasicCollection from '../../../collection/BasicCollection';
import PopupMenuView from '../PopupMenuView';
import cookie from 'bemmy/src/blocks/component/cookie';
import { setCookieWithExpire } from '../../../helpers/utils/cookie';

/**
 * @class AutoUpdateControlView
 * @extends PopupMenuView
 */
const AutoUpdateControlView = PopupMenuView.extend({

    className: 'autoupdate-control',

    template: require('./tpl/AutoUpdateControlView.hbs'),

    isActive: false,

    initialize() {
        PopupMenuView.prototype.initialize.apply(this, arguments);
        this.listenTo(Backbone, 'autoupdate:initialize', this.updateProps);
        this.listenTo(Backbone, 'autoupdate:ready', this.autoupdateReadyToInit);
    },

    ui: _.extend({}, PopupMenuView.prototype.ui, {
        menu: '.menu',
        trigger: '.auto_update_toggler',
        stopper: '.stop_updating',
        menuItems: '.menu__item'
    }),

    events: _.extend({}, PopupMenuView.prototype.events, {
        'click @ui.menu': 'toggleMenu',
        'click @ui.trigger': 'triggerSingleUpdate',
        'click @ui.stopper': 'stopUpdating',
        'click @ui.menuItems': 'triggerUpdating'
    }),

    serializeData() {
        return {
            size: (this.options.size ? this.options.size : 'M'),
            isActive: this.isActive,
            iconOnly: (this.isActive ? 'no' : 'yes'),
            triggerLabel: (this.isActive ? this.triggerLabel : '')
        };
    },

    autoupdateReadyToInit() {
        const self = this;

        setTimeout(() => {
            let interval = cookie.get('autoupdatePreference');

            if (interval) {
                interval = Number(interval);

                self.triggerUpdatingWithParsedValue(interval);
            }
        }, 0);
    },

    stopUpdating() {
        Backbone.trigger('autoupdate:off');

        setCookieWithExpire('autoupdatePreference', '', -1);

        this.isActive = false;
        this.render();
    },

    triggerUpdating(evtObj) {
        evtObj.preventDefault();

        const interval = parseInt(evtObj.target.dataset.updateInterval, 10);

        setCookieWithExpire('autoupdatePreference', interval, 365);

        this.triggerUpdatingWithParsedValue(interval);
    },

    triggerUpdatingWithParsedValue(parsedInterval) {
        if (parsedInterval) {
            Backbone.trigger('autoupdate:on', parsedInterval);

            this.updateProps(parsedInterval);
        }
    },

    updateProps(interval) {
        this.isActive = true;
        this.triggerLabel = (interval + 's');
        this.render();
    },

    triggerSingleUpdate() {
        Backbone.trigger('autoupdate:on', BasicCollection.ONCE);
    },

    onClose() {
        this.stopListening(Backbone, 'autoupdate:initialize', this.updateProps);
    }
});

module.exports = AutoUpdateControlView;
