/* global Notification */

/**
 * Mixin adds functionality for pushing notifications.
 *
 * @mixin
 */
const PushNotificationsMixin = {

    initializePushNotificationsClient() {
        if (window.Notification && Notification.permission !== 'granted') {
            const descriptor = Object.getOwnPropertyDescriptor(Notification, 'permission');

            if (descriptor && descriptor.writable) {
                Notification.requestPermission(status => {
                    if (Notification.permission !== status) {
                        Notification.permission = status;
                    }
                });
            }
        }

        this._isHidden = document.hidden === undefined ? false : document.hidden;

        document.addEventListener('visibilitychange', () => {
            this._isHidden = Boolean(document.hidden);
        });

        this.listenTo(Backbone, 'notifications:push', this.pushNotification);
    },

    _getTitle() {
        return 'Sandbox';
    },

    _getIcon() {
        return '/favicon-notify.png';
    },

    _getBody(task) {
        const description = task.get('description');

        return task.get('type') + ' #' + task.get('id') + ' is ' + task.get('status') +
            (description ? "\nDescription: " + description : ''); // eslint-disable-line
    },

    _getTag(task) {
        return '#' + task.get('id');
    },

    pushNotification(task, pageProps) {
        if (
            this._isHidden &&
            task &&
            pageProps &&
            window.Notification &&
            window.Notification.permission === 'granted'
        ) {
            const currentStatus = task.get('status');
            const affordableStatusForNotification = pageProps && currentStatus &&
                (pageProps.pushNotificationFor || [currentStatus]).includes(currentStatus);

            if (affordableStatusForNotification) {
                const title = this._getTitle();
                const options = {
                    body: this._getBody(task),
                    icon: this._getIcon(),
                    tag: this._getTag(task)
                };

                const n = new Notification(title, options); // eslint-disable-line
                n.onclick = function () {
                    window.focus();
                    setTimeout(n.close.bind(n), 150);
                };
            }
        }
    }
};

module.exports = PushNotificationsMixin;
