const TaskModel = require('../../../model/TaskModel');

/**
 * @class StatusSelectorView
 * @extends Backbone.Marionette.ItemView
 *
 * @property {Backbone.Model} model
 */
const StatusSelectorView = Marionette.ItemView.extend({

    className: 'status_selector_view',

    template: require('./tpl/StatusSelectorView.hbs'),

    ui: {
        cnt: '.button',
        label: '.button_mode_dropdown .button__label',
        check: '.check',
        allChecker: '.check_all'
    },

    events: {
        'bemmy:set @ui.check': 'onSetUpdated',
        'click @ui.allChecker': 'onAllSelect'
    },

    options: {
        selectAll: false,
        fullWidth: false,
        statusField: 'state',
        defaultStateTitle: 'No statuses are selected'
    },

    initialize(options) {
        this.options = _.extend({}, this.options, options);
        this.onSetUpdated = _.debounce(this.onSetUpdated, 50).bind(this);

        this.listenTo(this.model, 'change:' + this.options.statusField, this.mapModelToView);
    },

    onRender() {
        this.$('.button').bemmyButton();
        this.$('.check').bemmyCheck();
        this.$('.check-list').bemmyCheckList();
        this.updateTriggerTitle();

        if (this.options.fullWidth) {
            this.setContainerFullWidth();
        }
    },

    mapModelToView(model, value, options) {
        if (options && !options.fromStateSelector) {
            this.render();
        }
    },

    serializeData() {
        return {
            groups: this.serializeStatusGroups(),
            selectAll: this.options.selectAll,
            triggerLabel: this.options.defaultStateTitle
        };
    },

    serializeStatusGroups() {
        const serialized = [];
        const groups = TaskModel.getStatusGroups();
        const virtualGroups = TaskModel.getVirtualStatusGroups();
        const setStatuses = this.model.get(this.options.statusField);

        _.forEach(groups, (value, key) => {
            groups[key] = _.difference(value, virtualGroups[key] || []);
        });

        Object.keys(groups).forEach(groupName => {
            let groupSelected = 0;
            const group = {
                title: groupName,
                statuses: groups[groupName].map(status => {
                    const selected = (setStatuses && setStatuses.includes(status));

                    if (selected) {
                        groupSelected++;
                    }

                    return {
                        status,
                        selected: (selected || setStatuses.includes(groupName))
                    };
                })
            };

            group.selected = (groupSelected === group.statuses.length) || (setStatuses.includes(groupName));
            group.inter = (groupSelected > 0);

            serialized.push(group);
        });

        return serialized;
    },

    onSetUpdated() {
        let isGroupChecked = false;
        const statusList = [];

        this.ui.check.each((index, elt) => {
            /* eslint max-statements: [1, 11] */
            /* eslint complexity: [1, 6] */

            elt = $(elt);

            const text = $.trim(elt.text());
            const wrapper = elt.closest('.check-list__head');

            if (wrapper.length > 0) {
                isGroupChecked = false;
            }

            if (elt.hasClass('checked')) {
                if (!elt.hasClass('indeterminate')) {
                    if (wrapper.length > 0) {
                        isGroupChecked = true;
                        statusList.push(text);
                    } else if (!isGroupChecked) {
                        statusList.push(text);
                    }
                }
            }
        });

        this.model.set(this.options.statusField, statusList, {
            fromStateSelector: true
        });

        this.updateTriggerTitle();
    },

    onAllSelect(evtObj) {
        evtObj.preventDefault();

        this.model.set(this.options.statusField, _.flatten(_.values(TaskModel.getStatusGroups())));
    },

    updateTriggerTitle() {
        this.ui.label.text(this.getTriggerTitle());
    },

    getTriggerTitle() {
        const stateList = this.model.get(this.options.statusField);

        return (
            stateList.length > 0 ?
                stateList.join(', ') :
                this.options.defaultStateTitle
        );
    },

    setContainerFullWidth() {
        this.ui.cnt.css('width', '100%');
    },

    onClose() {
        this.stopListening(this.model);
    }
});

module.exports = StatusSelectorView;
