/**
 * @class TimeInterval
 * @extends Marionette.ItemView
 *
 * @property {Backbone.Model} model
 */
const TimeInterval = Marionette.ItemView.extend({

    template: require('./tpl/timeInterval.hbs'),

    ui: {
        intervalValue: '.input',
        intervalValueInput: '.input .input__input',
        intervalType: '.select select'
    },

    events: {
        'bemmy:refresh @ui.intervalValue': 'mapViewToModel',
        'blur @ui.intervalValueInput': 'onIntervalTypeBlur',
        'change @ui.intervalType': 'onIntervalTypeChange'
    },

    options: {
        modelSetter: null,
        modelGetter: null,
        controlWidth: '121',
        bindedModelKey: '',
        defaultInterval: 0,
        defaultIntervalType: 'seconds'
    },

    initialize(options) {
        this.options = _.extend({}, this.options, options);

        if (this.options.bindedModelKey) {
            this.listenTo(this.model, ('change:' + this.options.bindedModelKey), this.mapModelToView);
        }
    },

    onRender() {
        this.$('.input').bemmyInput();
        this.$('.select').bemmySelect();

        this.mapModelToView();
    },

    onIntervalTypeBlur() {
        const value = this.ui.intervalValueInput.val();
        const type = this.ui.intervalType.val();

        if (type === 'seconds' && value < 60) {
            this.ui.intervalValueInput.val(60);
            this.mapViewToModel();
        }
    },

    onIntervalTypeChange() {
        this.ui.intervalValueInput.val(this.convertIntervalValue(
            this.ui.intervalValueInput.val(),
            this.options.defaultIntervalType,
            this.ui.intervalType.val()
        ));

        this.options.defaultIntervalType = this.ui.intervalType.val();
    },

    mapModelToView() {
        const storedInterval = this.getInterval();

        this.ui.intervalValueInput.val(this.convertIntervalValue(
            storedInterval.interval,
            'seconds',
            this.ui.intervalType.val()
        ));
    },

    mapViewToModel() {
        const interval = this.ui.intervalValueInput.val();
        const type = this.ui.intervalType.val();

        if (interval !== '') {
            this.options.modelSetter.call(
                this.model,
                this.convertIntervalValue(interval, type)
            );
        }
    },

    /**
     * @param {number} interval
     * @param {string} [typeFrom]
     * @param {string} [typeTo]
     * @returns {number}
     */
    convertIntervalValue(interval, typeFrom, typeTo) {
        /* eslint complexity: [1, 9] */

        const periodReg = /^(days)|(hours)|(minutes)|(seconds)$/;

        typeTo = (_.isString(typeTo) && typeTo.match(periodReg)) ? typeTo : 'seconds';
        typeFrom = (_.isString(typeFrom) && typeFrom.match(periodReg)) ? typeFrom : 'seconds';
        interval = parseFloat(interval) || 0;

        const duration = moment.duration(interval, typeFrom);

        switch (typeTo) {
            case 'days': { return duration.asDays(); }
            case 'hours': { return duration.asHours(); }
            case 'minutes': { return duration.asMinutes(); }
            default: { return duration.asSeconds(); }
        }
    },

    getInterval() {
        const interval = this.options.modelGetter.call(this.model);

        return {
            interval: (interval ? interval : this.options.defaultInterval),
            type: 'seconds'
        };
    },

    serializeData() {
        return {
            controlWidth: this.options.controlWidth
        };
    },

    onClose() {
        this.stopListening(this.model);
    }
});

module.exports = TimeInterval;
