/* eslint no-redeclare: 0 */

const _ = require('underscore');
const error = require('../../../error');
// eslint-disable-next-line no-unused-vars
const notification = require('../../../notification');
const TaskTypeModel = require('../../../model/resources/TaskTypeModel');
const ClipboardClientMixin = require('../../components/clipboardClient/ClipboardClientMixin');
const ClientsBatchCommentsControl = require('./ClientsBatchCommentsControl');
const ClientsBatchActionConfirm = require('./ClientsBatchActionConfirm');
const hostCopyTrigger = require('../../admin/manageTaskTypes/tpl/HostCopyTrigger.hbs');

/**
 * @class ClientsBatchControlsView
 * @extends Backbone.Marionette.ItemView
 *
 * @property {Marionette.Region} batchCommentControl
 * @property {Backbone.Collection} collection
 */
const ClientsBatchControlsView = Marionette.Layout.mixin(ClipboardClientMixin).extend({

    className: 'section__ii section__ii_invert',

    template: require('./tpl/ClientsBatchControlsView.hbs'),

    // Will store selected items not to iterate each time through the all collection
    selectedItems: null,

    ui: {
        selectedQntCnt: '.selected_qnt',
        actionBtnsCnt: '.action_btn_cnt',
        selectNone: '.select_none',
        selectAll: '.select_all',

        reload: '.reload_btn',
        reboot: '.reboot_btn',
        shutdown: '.shutdown_btn',
        cleanup: '.cleanup_btn',
        shortifyGetterCnt: '.shortify-getter-cnt'
    },

    options: {
        activeClass: 'active',
        disableBtnClass: 'button_disable'
    },

    events: {
        'click @ui.selectNone': 'onSelectNone',
        'click @ui.selectAll': 'onSelectAll',
        'click @ui.reload': 'onReload',
        'click @ui.reboot': 'onReboot',
        'click @ui.shutdown': 'onShutDown',
        'click @ui.cleanup': 'onCleanup'
    },

    collectionEvents: {
        change: 'onSelected',
        sync: 'reInitClipboard',
        'batch:deselect': 'reInitCollection'
    },

    initialize() {
        this.reInitCollection();
        this.updateView = _.debounce(this.updateView, 10).bind(this);
    },

    onRender() {
        if (this.selectedItems.length > 0) {
            this.ui.actionBtnsCnt.addClass(this.options.activeClass);
        }

        if (this.options.user.isAdmin()) {
            this.batchCommentControl.show(new ClientsBatchCommentsControl({
                collection: this.selectedItems
            }));
        }

        this.reInitClipboard();
    },

    serializeData() {
        return {
            isAdmin: (this.options.user.isAdmin()),
            selected: this.selectedItems.length,
            clip: true,
            total: this.collection.length,
            hosts: ''
        };
    },

    onSelected(model) {
        if (model.get('selected')) {
            this.selectedItems.add(model);
        } else {
            this.selectedItems.remove(model);
        }

        this.updateView();
    },

    onSelectNone() {
        while (this.selectedItems.models.length) {
            _.map(this.selectedItems.models, client => {
                if (client) {
                    client.set({ selected: false }, { source: 'controls' });
                }
            });
        }
    },

    onSelectAll() {
        this.collection.each(client => {
            if (client) {
                client.set({ selected: true }, { source: 'controls' });
            }
        });
    },

    onReload() {
        this.onBatchAction('reload');
    },

    onReboot() {
        this.onBatchAction('reboot');
    },

    onShutDown() {
        this.onBatchAction('shutdown');
    },

    onCleanup() {
        this.onBatchAction('cleanup');
    },

    onBatchAction(action) {
        if (!this.options.user.isAdmin()) {
            return false;
        }

        (new ClientsBatchActionConfirm({
            action,
            selectedItems: this.selectedItems.map(client => {
                return client.get('id');
            })
        })).show();
    },

    updateView() {
        this.ui.selectedQntCnt.text(this.selectedItems.length);
        this.ui.actionBtnsCnt.toggleClass(
            this.options.activeClass,
            Boolean(this.selectedItems.length)
        );
        this.ui.selectNone.toggleClass(
            this.options.disableBtnClass,
            this.selectedItems.length === 0
        );
        this.ui.selectAll.toggleClass(
            this.options.disableBtnClass,
            this.selectedItems.length === this.collection.length
        );
    },

    reInitCollection() {
        this.selectedItems = new Backbone.Collection(this.collection.filter(client => {
            return client.get('selected');
        }));
    },

    reInitClipboard() {
        const self = this;
        const typeModel = new TaskTypeModel({});

        typeModel
            .fetchShortenedHosts(this.collection.map(client => {
                return client.attributes.id;
            }))
            .done(shortened => {
                if (self.ui && self.ui.shortifyGetterCnt && self.ui.shortifyGetterCnt.html) {
                    self.ui.shortifyGetterCnt.html(hostCopyTrigger({
                        toCopy: shortened,
                        label: 'Get shortened clients',
                        clip: true
                    }));
                }

                self.initClipboardClient();
            })
            .fail(() => {
                error.message('Problem with getting shortened hosts');
            });
    },

    initClipboardClient() {
        this.initializeClipboardClient(
            '.copy_list_trigger',
            'Hosts list has been copied to clipboard!'
        );
    }
});

module.exports = ClientsBatchControlsView;
