import React from 'react';
import ReactDOM from 'react-dom';

/* eslint max-statements: [1, 15] */

import FilterPresetSavePromptView from './presetSaveConfirm/FilterPresetSavePromptView';
import TaskTypeAutocompleteView from './selects/autocomplete/TaskTypeAutocompleteView';
import DateTimeIntervalView from './selects/datetime/DateTimeIntervalView';
import FormAcceptBehavior from '../../behaviors/forms/FormAcceptBehavior';
import StatusSelectorView from '../../components/statusSelector/StatusSelectorView';
import AuthorSuggestView from './suggest/AuthorSuggestView';
import AutocompleteView from './selects/autocomplete/AutocompleteView';
import OwnerSuggestView from './suggest/OwnerSuggestView';
// eslint-disable-next-line no-unused-vars
import FilterModel from '../../../model/filter/FilterModel';
import app from '../../../app';

import { SuggestTagMulti } from '../../../components/SuggestTag';
import ParameterInput from '../../../components/ParameterInput';
import HintsInput from '../../../components/HintsInput';
import i18n from '../../../components/i18n';

/**
 * @class FilterLayout
 * @extends Marionette.Layout
 *
 * @property {Object}          options
 * @property {Object}          options.taskProps
 * @property {FilterModel}     model
 * @property {TasksCollection} collection
 * @property {Marionette.Region} createInterval
 * @property {Marionette.Region} updateInterval
 * @property {Marionette.Region} typeSelector
 * @property {Marionette.Region} hostSelector
 * @property {Marionette.Region} ownerSuggest
 * @property {Marionette.Region} authorSuggest
 * @property {Marionette.Region} statusSelector
 */
const FilterLayout = Marionette.Layout.extend({
    tagName: 'form',
    className: 'task_list_filter',
    template: require('./tpl/FilterLayout.hbs'),

    /* REGIONS DEFINED DIRECTLY IN THE TEMPLATE */

    options: {
        extendedToggleClass: 'b_hidden',
        triggerActiveClass: 'button_type_hl',
        isOpen: false
    },

    ui: {
        // Filter controls and area containers
        favToggler: '.button_fav',
        filterReset: '.app_filter_reset',
        filterTrigger: '.app_filter_trigger',
        extendedAreas: '.b_toggle',
        extendedToggler: '.button_filter-show',

        // Filter inputs
        priority: '.app_filter_priority',
        arch: '.app_filter_arch',
        imprt: '.app_filter_imprt',
        resId: '.app_filter_rid input',
        descr: '.app_filter_descr',
        descrInput: '.app_filter_descr input',
        hidden: '.app_filter_hidden',
        children: '.app_filter_children',

        tag: '.app_filter_tag',
        hints: '.app_filter_hint',
        inputParameters: '.app_filter_input_parameters',
        outputParameters: '.app_filter_output_parameters'
    },

    events: {
        // Filter controls and area containers
        'click @ui.extendedToggler': 'onExtendedToggle',
        'click @ui.favToggler': 'onFilterSave',
        'click @ui.filterTrigger': 'onFilterTrigger',
        'click @ui.filterReset': 'onFilterReset',

        // Filter inputs
        'change @ui.resId': 'onResIdSet',
        'bemmy:refresh @ui.descr': 'onDescrSet',
        'bemmy:set @ui.priority': 'onPrioritySet',
        'bemmy:set @ui.arch': 'onArchSet',
        'bemmy:set @ui.imprt': 'onImprtSet',
        'bemmy:set @ui.hidden': 'onHiddenSet',
        'bemmy:set @ui.children': 'onChildrenSet',

        'changed:interval': 'onValuesChanged'
    },

    modelEvents: {
        change: 'onValuesChanged',
        'change:created': 'showCreatedRegion',
        'change:updated': 'showUpdatedRegion',
        blank: 'render',
        'change:tags': 'renderControls',
        'change:all_tags': 'renderBlankControls',
        'change:hints': 'renderBlankControls',
        'change:all_hints': 'renderBlankControls',
        'change:any_params': 'renderBlankControls',
        'change:input_parameters': 'renderBlankControls',
        'change:output_parameters': 'renderBlankControls'
    },

    behaviors: {
        FormAcceptBehavior: {}
    },

    initialize() {
        const isOpen = window.localStorage &&
            window.localStorage.getItem &&
            window.localStorage.getItem('TASKLIST_FILTER_STATE_OPEN') !== 'false';

        this.listenTo(this, FormAcceptBehavior.EVENTS.ACCEPT_EVENT, this.onFormAccept);

        if (isOpen === undefined) {
            this.isAnyExtraFieldsFilled();
        } else {
            this.options.isOpen = isOpen;
        }

        this.handleHintsSet = this.handleHintsSet.bind(this);
        this.handleHintsAllSet = this.handleHintsAllSet.bind(this);
        this.handleTagsSet = this.handleTagsSet.bind(this);
        this.handleTagsAllSet = this.handleTagsAllSet.bind(this);
        this.handleInputParameterChange = this.handleInputParameterChange.bind(this);
        this.handleOutputParameterChange = this.handleOutputParameterChange.bind(this);
        this.handleInputParameterToggleChange = this.handleInputParameterToggleChange.bind(this);
        this.handleOutputParameterToggleChange = this.handleOutputParameterToggleChange.bind(this);
    },

    onRender() {
        this.$('.input').bemmyInput();
        this.$('.check').bemmyCheck();
        this.$('.select').bemmySelect();
        this.$('.button').bemmyButton();

        this.renderControls();

        this.showRegions();

        this.$el.validate({
            onChange: true,
            eachValidField() {
                $(this).removeClass('not-valid');
            },
            eachInvalidField() {
                $(this).addClass('not-valid');
            }
        });

        if (this.options.isOpen) {
            this.$el.addClass('filter_extended');
        }

        // To prevent highlighting «Apply» button on default values set
        const self = this;

        setTimeout(() => {
            if (self.ui.filterTrigger && !_.isString(self.ui.filterTrigger)) {
                self.ui.filterTrigger.removeClass(self.options.triggerActiveClass);
            }
        }, 0);
    },

    renderBlankControls() {
        this.renderControls();
    },

    renderControls(model, tags) {
        const _tags = this.model.get('tags');
        const togglerValue = Boolean(this.model.get('all_tags'));
        const paramsToggler = this.model.get('any_params') === null ? false : this.model.get('any_params');
        const inputParams = this.model.get('input_parameters');
        const outputParams = this.model.get('output_parameters');
        const hints = this.model.get('hints');
        const hintsToggler = Boolean(this.model.get('all_hints'));

        if (this.ui.tag[0]) {
            ReactDOM.render(
                <SuggestTagMulti
                    value={tags || (_tags instanceof Array ? _tags : _tags.split(','))}
                    togglerValue={togglerValue}
                    name={'tags'}
                    placeholder={i18n.labels.tags}
                    onChange={this.handleTagsSet}
                    onToggle={this.handleTagsAllSet}
                    />,
                this.ui.tag[0]
            );
        }

        if (this.ui.hints[0]) {
            ReactDOM.render(
                <HintsInput
                    value={hints instanceof Array ? hints : hints.split(',')}
                    togglerValue={hintsToggler}
                    placeholder="Task hints"
                    onChange={this.handleHintsSet}
                    onToggle={this.handleHintsAllSet}
                    />,
                this.ui.hints[0]
            );
        }

        if (this.ui.inputParameters[0]) {
            ReactDOM.render(
                <ParameterInput
                    value={inputParams}
                    placeholder="Input parameters"
                    togglerValue={paramsToggler}
                    onChange={this.handleInputParameterChange}
                    onToggle={this.handleInputParameterToggleChange}
                    />,
                this.ui.inputParameters[0]
            );
        }

        if (this.ui.outputParameters[0]) {
            ReactDOM.render(
                <ParameterInput
                    value={outputParams}
                    placeholder="Output parameters"
                    togglerValue={paramsToggler}
                    onChange={this.handleOutputParameterChange}
                    onToggle={this.handleOutputParameterToggleChange}
                    />,
                this.ui.outputParameters[0]
            );
        }
    },

    showRegions() {
        this.ownerSuggest.close();
        this.typeSelector.close();
        this.hostSelector.close();
        this.authorSuggest.close();
        this.statusSelector.close();

        this.ownerSuggest.show(this.getOwnerSuggest());
        this.typeSelector.show(this.getTypeSelector());
        this.hostSelector.show(this.getHostSelector());
        this.authorSuggest.show(this.getAuthorSuggest());
        this.statusSelector.show(this.getStatusSelector());

        this.showCreatedRegion();
        this.showUpdatedRegion();
    },

    showUpdatedRegion() {
        this.updateInterval.close();
        this.updateInterval.show(this.getUpdateIntervalRegion());
    },

    showCreatedRegion() {
        this.createInterval.close();
        this.createInterval.show(this.getCreateIntervalRegion());
    },

    handleHintsSet(hints) {
        this.model.set('hints', hints);
    },

    handleHintsAllSet(flag) {
        this.model.set('all_hints', flag);
    },

    handleTagsSet(tags) {
        this.model.set('tags', tags);
    },

    handleTagsAllSet(flag) {
        this.model.set('all_tags', flag);
    },

    handleInputParameterChange(value) {
        this.model.set('input_parameters', value);
    },

    handleInputParameterToggleChange(value) {
        this.model.set('any_params', value);
    },

    handleOutputParameterChange(value) {
        this.model.set('output_parameters', value);
    },

    handleOutputParameterToggleChange(value) {
        this.model.set('any_params', value);
    },

    onExtendedToggle() {
        if (window.localStorage && window.localStorage.setItem) {
            window.localStorage.setItem('TASKLIST_FILTER_STATE_OPEN', !this.options.isOpen);
        }

        this.$el.toggleClass('filter_extended', !this.options.isOpen);
        this.ui.extendedAreas.toggleClass(this.options.extendedToggleClass, this.options.isOpen);
        this.options.isOpen = !this.options.isOpen;
    },

    onFilterSave(evtObj) {
        evtObj.preventDefault();

        (new FilterPresetSavePromptView({
            model: this.model.clone(),
            collection: this.options.user.getPresets()
        })).show();
    },

    onFilterTrigger(evtObj) {
        if (evtObj) {
            evtObj.preventDefault();
        }

        const filterValues = this.model.clone();

        this.ui.filterTrigger.removeClass(this.options.triggerActiveClass);

        app.trigger('change:filterparams:preset', filterValues);
        this.updateLastFilter(filterValues);
    },

    onFilterReset() {
        this.model.resetFilter();

        return false;
    },

    onImprtSet(evtObj, value) {
        this.model.set('important', value);
    },

    onResIdSet(evtObj) {
        this.model.set('resource_id', evtObj.target.value);
    },

    onDescrSet() {
        this.model.set('desc_re', this.ui.descrInput.val());
        this.ui.descrInput.change();
    },

    onHiddenSet(evtObj, value) {
        this.model.set('hidden', value);
    },

    onChildrenSet(evtObj, value) {
        this.model.set('children', value);
    },

    onArchSet(evtObj, value) {
        this.setSelectedValue(evtObj, value, 'arch');
    },

    onPrioritySet(evtObj, value) {
        this.setSelectedValue(evtObj, value, 'priority');
    },

    onValuesChanged() {
        this.ui.filterTrigger.addClass(this.options.triggerActiveClass);
    },

    setSelectedValue(evtObj, value, propName) {
        evtObj.preventDefault();
        evtObj.stopPropagation(); // What the fuck is magic here? •_•

        this.model.set(propName, value);
    },

    getCreateIntervalRegion() {
        return new DateTimeIntervalView({
            model: this.model.get('created'),
            presets: [
                { key: '3_days', label: 'Last 3 days' },
                { key: '7_days', label: 'Last 1 week' },
                { key: '14_days', label: 'Last 2 weeks' },
                { key: '1_months', label: 'Last month' }
            ],
            placeholder: 'Created',
            classNamePrefix: 'app_filter_create'
        });
    },

    /**
     * @description
     * @returns {DateTimeIntervalView}
     */
    getUpdateIntervalRegion() {
        return new DateTimeIntervalView({
            model: this.model.get('updated'),
            presets: [
                { key: '5_minutes', label: 'Last 5 minutes' },
                { key: '15_minutes', label: 'Last 15 minutes' },
                { key: '30_minutes', label: 'Last 30 minutes' }
            ],
            dateOnly: false,
            placeholder: 'Updated',
            classNamePrefix: 'app_filter_update'
        });
    },

    getTypeSelector() {
        return new TaskTypeAutocompleteView({
            lazy: 50,
            model: this.model,
            items: this.options.taskProps.types,
            value: this.model.get('type'),
            placeHolder: 'Task type',
            showAnyItem: true,
            modelTargetKey: 'type',
            selectClassName: 'app_filter_type'
        });
    },

    getHostSelector() {
        return new AutocompleteView({
            lazy: 70,
            model: this.model,
            items: this.options.taskProps.hosts,
            value: this.model.get('host'),
            placeHolder: 'Host',
            showAnyItem: true,
            modelTargetKey: 'host',
            selectClassName: 'app_filter_host'
        });
    },

    getOwnerSuggest() {
        return new OwnerSuggestView({
            model: this.model,
            ownerGroups: this.options.taskProps.groups.convertToList(),
            defaultValue: this.model.get('owner')
        });
    },

    getAuthorSuggest() {
        return new AuthorSuggestView({
            model: this.model,
            defaultValue: this.model.get('author')
        });
    },

    getStatusSelector() {
        return new StatusSelectorView({
            model: this.model,
            fullWidth: true,
            statusField: 'status'
        });
    },

    serializeData() {
        return _.extend(
            this.model.toJSON(),
            this.options.taskProps,
            {
                isOpen: this.options.isOpen,
                created: this.model.get('created').serialize(),
                updated: this.model.get('updated').serialize(),
                toggleClass: this.options.extendedToggleClass
            }
        );
    },

    updateLastFilter(filterValues) {
        const defaultFilter = app.request('DEFAULT_FILTER');

        defaultFilter.taskList.set(filterValues);
        defaultFilter.taskList.saveFilterState();
    },

    onFormAccept() {
        const self = this;

        setTimeout(() => {
            self.onFilterTrigger();
        }, 0);
    },

    isAnyExtraFieldsFilled() {
        const filterValues = this.model.serializeAsUrl();
        const extraFilterKeys = [
            'children',
            'created',
            'updated',
            'desc_re',
            'hidden',
            'model',
            'owner',
            'host',
            'arch',
            'input_parameters',
            'output_parameters'
        ];

        return (_.intersection(Object.keys(filterValues), extraFilterKeys) || []).length && true;
    }
});

FilterLayout.DATE_FORMAT = 'DD.MM.YYYY';
FilterLayout.IS_CUSTOM_INTERVAL = 'cstm';
FilterLayout.DEFAULT_TIME_MEASURE = 'days';

module.exports = FilterLayout;
