const QuickModel = require('../../../../../model/QuickModel');
const lazyExecutor = require('../../../../../helpers/utils/lazyExecutor');
const layoutMapper = require('../../../../mixin/layoutMapper');
const suggestMixin = require('../../../../mixin/suggestMixin');
const QuickCompositeView = require('../../../../utils/QuickCompositeView');

/**
 * @class AutocompleteView
 * @extends Marionette.CompositeView
 * @mixes layoutMapper
 *
 * @property {Backbone.Model} model - In most of cases that will be a filter model.
 */
const AutocompleteView = QuickCompositeView.mixin(layoutMapper, suggestMixin).extend({

    template: require('./tpl/AutocompleteView.hbs'),

    className: 'autocomplete_cnt',

    itemView: require('./AutocompleteItemView'),
    itemViewContainer: '.menu__group',

    options: {
        lazy: 0,
        items: [],
        value: '',
        placeHolder: '',
        hiddenIndexes: [],
        selectClassName: '',
        emptyVal: '',
        showAnyItem: false,
        anyItemLabel: 'Any',
        modelTargetKey: ''
    },

    behaviors: {
        SuggestNavigationBehavior: {},
        ClickOut: {
            eventName: 'clickOut',
            includeSelf: false
        }
    },

    ui: {
        items: 'li',
        input: '.input__input',
        select: '.select',
        itemsCnt: '.menu__group',
        toggler: '.input__arrow'
    },

    events: {
        'focus @ui.input': 'onInputFocus',
        'blur @ui.input': 'onInputBlur',
        'keyup @ui.input': 'onKeyUp',
        'click @ui.toggler': 'onMenuToggle'
    },

    initialize(options) {
        this.options = _.extend({}, this.options, options);

        if (this.options.lazy) {
            this.stopListening(this.collection, 'add', this.addChildView);
            this.stopListening(this.collection, 'remove', this.removeChildView);
        }

        this.createCollectionFromItems();
        this.onKeyUp = _.debounce(this._onKeyUp, 120).bind(this);

        this.on('selected', this.onItemSelect);
        this.listenTo(this.model, 'change:' + this.options.modelTargetKey, this.mapModelToView);
    },

    onMenuToggle() {
        const isOpen = this.ui.select.hasClass('select_open');

        this.toggleMenu(!isOpen);

        if (!isOpen) {
            this.ui.input.focus();
        }
    },

    serializeData() {
        return {
            items: this.options.items,
            value: this.options.value,
            emptyVal: this.options.emptyVal,
            placeHolder: this.options.placeHolder,
            selectClassName: this.options.selectClassName
        };
    },

    _getSearchValue() {
        const value = this.ui.input.val();

        if (value.length > 0 && value.charCodeAt(0) > 255) {
            return this._convertToEngLayout(value.toLowerCase());
        }

        return value;
    },

    _onKeyUp(evtObj) {
        if (!this.isControlKey(evtObj.keyCode)) {
            const self = this;
            const searchValue = this._getSearchValue();

            if (searchValue === '') {
                this.onItemSelect('');
                this.setDefaultFilter();
            } else {
                this.setFilter(item => {
                    const itemLabel = item.get('label');

                    return (new RegExp(searchValue, 'i')).test(itemLabel) ||
                        itemLabel === self.options.anyItemLabel;
                });
            }

            this._renderChildren();
            this.toggleMenu(Boolean(this.collection.length));
        }
    },

    onInputFocus() {
        if (!this.ui.select.hasClass('select_open')) {
            this.onMenuToggle();
        }
    },

    onInputBlur() {
        const searchValue = this._getSearchValue();

        this.onItemSelect({ value: searchValue }, true, true);
    },

    toggleMenu(targetState) {
        this.ui.select.toggleClass('select_open', targetState);
        this.ui.toggler.toggleClass('button_open', targetState);
    },

    onCompositeCollectionBeforeRender() {
        this.ui.itemsCnt.html('');
    },

    onCompositeCollectionRendered() {
        /** {@link SuggestNavigationBehavior} will update cached set of items on that event */
        this.collection.trigger('change:items');
    },

    createCollectionFromItems() {
        const self = this;

        this.collection = new Backbone.Collection(null, {
            model: QuickModel
        });

        if (this.options.lazy) {
            lazyExecutor.addTask(() => {
                self.mapItemsToCollection();
            }, this.options.lazy);
        } else {
            this.mapItemsToCollection();
        }
    },

    mapItemsToCollection() {
        const items = this.options.items.map(function (item) {
            return {
                id: item,
                label: this._getKeyOrValue(item, 'key'),
                value: this._getKeyOrValue(item, 'value')
            };
        }, this);

        if (this.options.showAnyItem) {
            items.unshift({
                id: this.options.anyItemLabel,
                label: this.options.anyItemLabel,
                value: this.options.anyItemLabel
            });
        }

        this.collection.reset(items, {
            silent: true
        });
    },

    /**
     * Will set value to model target key. Map it to the autocomplete and toggle the menu if necessary.
     *
     * @memberOf AutocompleteView
     *
     * @param {string}  selectedValue
     * @param {boolean} toggle
     */
    onItemSelect(selectedValue, toggle) {
        toggle = toggle || false;

        const value = this._getKeyValueOf(selectedValue.value, 'value');
        const valueToSet = (selectedValue.value === this.options.anyItemLabel ? '' : value);

        this.model.set(
            this.options.modelTargetKey,
            valueToSet
        );

        this.toggleMenu(toggle);
    },

    onClickOut() {
        this.toggleMenu(false);
    },

    setValue(value) {
        this.ui.input.val(value).change();
    },

    mapModelToView() {
        const target = this.model.get(this.options.modelTargetKey);
        const value = this._getKeyValueOf(target, 'key');

        this.setValue(value);
    },

    isControlKey(keyCode) {
        const controlKeys = [
            CONST.KEYS.ENTER,
            CONST.KEYS.TAB,
            CONST.KEYS.UP,
            CONST.KEYS.DOWN,
            CONST.KEYS.LEFT,
            CONST.KEYS.RIGHT,
            CONST.KEYS.SHIFT,
            CONST.KEYS.CTRL,
            CONST.KEYS.ALT,
            CONST.KEYS.CMD_L,
            CONST.KEYS.CMD_R
        ];

        return controlKeys.includes(keyCode);
    },

    onClose() {
        this.stopListening(this, 'selected');
        this.stopListening(this.model, 'change:' + this.options.modelTargetKey);
    }
});

module.exports = AutocompleteView;
