const fuzzy = require('fuzzy.js');
const lazyDash = require('../../../../../helpers/utils/lazyDash');
const AutocompleteView = require('./AutocompleteView');
// eslint-disable-next-line no-unused-vars
const AutocompleteItemsCollection = require('../../../../../collection/AutocompleteItemsCollection');

/**
 * @class TaskTypeAutocompleteView
 * @extends AutocompleteView
 */
const TaskTypeAutocompleteView = AutocompleteView.extend({

    itemView: require('./TaskTypeAutocompleteItemView'),
    itemViewOptions: {},
    suggestLimit: 11, // Increased as «Any» item added,

    options: _.extend({}, AutocompleteView.prototype.options, {
        fuzzy: true,
        langTransmit: true
    }),

    mapItemsToCollection() {
        const items = this.options.items.map(item => {
            return {
                id: item.attributes.type,
                label: item.attributes.type,
                description: item.attributes.description
            };
        }, this);

        if (this.options.showAnyItem) {
            items.unshift({
                id: this.options.anyItemLabel,
                label: this.options.anyItemLabel
            });
        }

        this.collection.reset(items, {
            silent: true
        });
    },

    _onKeyUp(evtObj) {
        if (!this.isControlKey(evtObj.keyCode)) {
            this.toggleMenu(true);

            const searchValue = this._getSearchValue();

            this.itemViewOptions.searchValue = searchValue;

            if (searchValue === '') {
                this.setDefaultFilter();
            } else {
                this.setFilter(TaskTypeAutocompleteView.getTaskTypeAutocompleteFilter(
                    searchValue,
                    this.suggestLimit
                ));
                this.toggleMenu(Boolean(this.collection.length));
            }

            this._renderChildren();
        }
    },

    _realRenderChildren() {
        const _this = this;

        if (this._isRenderChildren) {
            return;
        }

        this._isRenderChildren = true;

        this.startBuffering();

        this.closeEmptyView();
        this.closeChildren();

        lazyDash(this.collection)
            .filter((item, index) => {
                return _this._filter(item, index);
            })
            .sortBy(item => {
                if (_this._filter.matchedIds && _this._filter.matchedIds.length) {
                    return _this._filter.matchedIds.indexOf(item.get('id'));
                }

                return item.get('id');
            })
            .forEach((item, index) => {
                _this.addItemView(item, _this.getItemView(item), index);
            })
            .then(() => {
                _this.endBuffering();
                _this._isRenderChildren = false;
                _this.collection.trigger('change:items');
            });
    }
}, {

    getTaskTypeAutocompleteFilter(searchValue, suggestLimit) {
        const filter = function (item) {
            if (!filter.matchedIds) {
                const matched = [];

                for (let q = 0; q < item.collection.length; q++) {
                    matched.push(fuzzy(item.collection.models[q].get('label'), searchValue));
                    matched[matched.length - 1].model = item.collection.models[q];
                }

                filter.matchedIds = matched
                    .sort(fuzzy.matchComparator)
                    .filter(match => {
                        return match.score > 0;
                    })
                    .slice(0, suggestLimit)
                    .sort((a, b) => {
                        const ai = a.term.toLowerCase().indexOf(a.query.toLowerCase());
                        const bi = b.term.toLowerCase().indexOf(b.query.toLowerCase());

                        return (a.score === b.score) ?
                            bi < ai :
                            b.score - a.score;
                    })
                    .map(match => {
                        return match.model.get('id');
                    });
            }

            return (filter.matchedIds.includes(item.get('id')));
        };

        return filter;
    }
});

module.exports = TaskTypeAutocompleteView;
