const DateTimeIntervalModel = require('../../../../../model/DateTimeIntervalModel');

/**
 * @class DateTimeIntervalView
 * @extends Backbone.Marionette.ItemView
 *
 * @property {Object}   options
 * @property {Object[]} options.presets          - Describes options for select (in select mode) and dd menu
 *                                                 (in custom period mode). Each item is a hash with props «label»
 *                                                 and «key» that represents text and keys for select/menu. Each
 *                                                 «key» should match %value%_%measure% format (like: 5_minutes,
 *                                                 7_days), supportable measures are: «days», «months», «minutes»,
 *                                                 «hours».
 * @property {Boolean}  options.dateOnly         - Will operate with only dates in the datepickers.
 * @property {String}   options.placeholder      - Placeholder for the fields
 * @property {Boolean}  options.customPeriod     - Flag that indicates if customPeriod mode is supported
 * @property {String}   options.classNamePrefix
 * @property {Boolean}  options.customPeriodMode - Flag based on which view will be rendered
 * @property {DateTimeIntervalModel} model
 */
const DateTimeIntervalView = Marionette.ItemView.extend(/** @lends DateTimeIntervalView.prototype */{

    className: 'b b_table',

    template: require('./tpl/DateTimeIntervalView.hbs'),

    options: {
        presets: [],
        dateOnly: true,
        placeholder: '',
        customPeriod: true,
        classNamePrefix: '',
        customPeriodMode: false
    },

    /* UI elements binded in #bindUIElements */

    initialize(options) {
        this.options = _.extend({}, this.options, options);
        this.model.options.dateOnly = this.options.dateOnly;

        if (this.model.get('offset') === DateTimeIntervalModel.CSTM) {
            this.options.customPeriodMode = true;
        }
    },

    onRender() {
        this.$('.button').bemmyButton();
        this.$('.select').bemmySelect();
        this.$('.input_date').bemmyDatepicker(null, { syncOnBlur: this.options.dateOnly });

        this.ui.switcher.on('click', this.onMenuPeriodChange.bind(this));
        this.ui.select.on('change', this.onSelectPeriodChange.bind(this));
        this.ui.from.on('input', _.debounce(this.onFromManualChange, 100).bind(this));
        this.ui.till.on('input', _.debounce(this.onTillManualChange, 100).bind(this));

        if (this.options.customPeriodMode) {
            this.listenTo(this.ui.from.bemmy(), 'date', this._setFrom);
            this.listenTo(this.ui.till.bemmy(), 'date', this._setTill);
        }
    },

    bindUIElements() {
        this.ui = {
            from: this.$('.' + this.options.classNamePrefix + '_from'),
            till: this.$('.' + this.options.classNamePrefix + '_till'),
            select: this.$('select'),
            switcher: this.$('.' + this.options.classNamePrefix + '_period')
        };
    },

    serializeData() {
        /* eslint max-statements: [1, 13] */

        const _serialized = this.model.toJSON();

        if (this.options.customPeriodMode) {
            _serialized.from = this._formatDate(_serialized.from);
            _serialized.till = this._formatDate(_serialized.till);
        }

        _serialized.presets = this.options.presets;
        _serialized.dateOnly = this.options.dateOnly;
        _serialized.fromClass = (this.options.classNamePrefix + '_from');
        _serialized.tillClass = (this.options.classNamePrefix + '_till');
        _serialized.placeholder = this.options.placeholder;
        _serialized.periodClass = (this.options.classNamePrefix + '_period');
        _serialized.customPeriodMode = this.options.customPeriodMode;
        _serialized.needCustomPeriod = this.options.customPeriod;

        return _serialized;
    },

    onPeriodChange(stringPeriod) {
        this.model.setDTInterval(null, null, stringPeriod);

        if (this.options.customPeriodMode) {
            if (stringPeriod !== DateTimeIntervalModel.CSTM) {
                this.options.customPeriodMode = false;
                this.render();
            }
        } else if (stringPeriod === DateTimeIntervalModel.CSTM) {
            this.options.customPeriodMode = true;
            this.render();
        }
    },

    onSelectPeriodChange(evtObj) {
        this.$el.trigger('changed:interval');
        this.onPeriodChange(evtObj.target.value);
    },

    onMenuPeriodChange(evtObj) {
        evtObj.preventDefault();
        this.$el.trigger('changed:interval');

        if (evtObj.target.classList.contains('menu__item')) {
            $(evtObj.currentTarget).data('bemmy').toggleHide();
            this.onPeriodChange($.trim(evtObj.target.className.replace('menu__item', '')));
        }
    },

    onFromManualChange(evtObj) {
        if (evtObj.target.value) {
            this._setFrom(evtObj.target.value);
        }
    },

    onTillManualChange(evtObj) {
        if (evtObj.target.value) {
            this._setTill(evtObj.target.value);
        }
    },

    _getFormat() {
        return this.options.dateOnly ?
            DateTimeIntervalView.DATE_FORMAT :
            DateTimeIntervalView.DATETIME_FORMAT;
    },

    _formatDate(dateToFormat) {
        const format = this._getFormat();

        return (dateToFormat === null ? '' : dateToFormat.format(format));
    },

    _setFrom(date) {
        this.$el.trigger('changed:interval');
        this._setDate('from', date);
    },

    _setTill(date) {
        this.$el.trigger('changed:interval');
        this._setDate('till', date);
    },

    _setDate(targetProp, date) {
        let target = null;
        const getter = ('get' + targetProp.substring(0, 1).toUpperCase() + targetProp.substring(1));

        if (date) {
            target = moment(date, this._getFormat());

            if (target.isValid()) {
                this.model.set(targetProp, target);
            }
        } else {
            target = this.model[getter](targetProp);
            this.ui[targetProp].bemmy().$input().val(this._formatDate(target));
        }
    },

    onClose() {
        this.stopListening(this.ui.switcher, 'click');
        this.stopListening(this.ui.select, 'change');
        this.stopListening(this.ui.from, 'input');
        this.stopListening(this.ui.from.bemmy());
    }
});

DateTimeIntervalView.IS_CUSTOM_INTERVAL = 'cstm';
DateTimeIntervalView.DATE_FORMAT = 'DD.MM.YYYY';
DateTimeIntervalView.TIME_FORMAT = 'HH:mm:ss';
DateTimeIntervalView.DATETIME_FORMAT = (DateTimeIntervalView.DATE_FORMAT + ' ' + DateTimeIntervalView.TIME_FORMAT);

module.exports = DateTimeIntervalView;
